# Entity Reference Autocomplete Add More

A Drupal module that provides an entity reference autocomplete form element with add/remove functionality for building dynamic forms.

## Description

This module provides the ability to add reference entities in custom forms with individual inputs that can be dynamically added or removed. It extends the standard entity autocomplete functionality with "Add another item" and "Remove" buttons, making it perfect for forms where users need to select multiple entities dynamically.

## Requirements

- Drupal 10 or 11
- Core Entity Reference module

## Installation

1. Install the module using composer:
   ```bash
   composer require drupal/entity_reference_autocomplete_add_more
   composer install
   ```

2. Enable the module:
   ```bash
   drush en entity_reference_autocomplete_add_more
   ```
   
   Or enable it through the admin interface at `/admin/modules`.

## Usage

### Basic Implementation

To use the Entity Reference Autocomplete Add More element in your custom forms, use the `entity_reference_autocomplete_add_more` form element type:

```php
$form['my_entities'] = [
  '#type' => 'entity_reference_autocomplete_add_more',
  '#title' => t('Select Entities'),
  '#target_type' => 'node', // The entity type to reference
  '#default_value' => [], // Array of default entity objects
  '#required' => FALSE,
  '#selection_settings' => [
    'target_bundles' => ['article', 'page'], // Optional: limit to specific bundles
  ],
];
```

### Form Element Properties

| Property | Type | Description | Required |
|----------|------|-------------|----------|
| `#type` | string | Must be `entity_reference_autocomplete_add_more` | Yes |
| `#title` | string | The label for the form element | No |
| `#target_type` | string | The entity type to reference (e.g., 'node', 'user', 'taxonomy_term') | Yes |
| `#default_value` | array | Array of default entity objects | No |
| `#required` | boolean | Whether at least one entity must be selected | No |
| `#selection_settings` | array | Additional selection criteria (bundles, etc.) | No |

### Advanced Configuration

#### Limiting to Specific Bundles

```php
$form['my_articles'] = [
  '#type' => 'entity_reference_autocomplete_add_more',
  '#title' => t('Select Articles'),
  '#target_type' => 'node',
  '#selection_settings' => [
    'target_bundles' => ['article'],
    'sort' => [
      'field' => 'created',
      'direction' => 'DESC',
    ],
  ],
];
```

#### With Default Values

```php
// Load some default entities
$default_nodes = \Drupal::entityTypeManager()
  ->getStorage('node')
  ->loadMultiple([1, 2, 3]);

$form['my_entities'] = [
  '#type' => 'entity_reference_autocomplete_add_more',
  '#title' => t('Select Content'),
  '#target_type' => 'node',
  '#default_value' => array_values($default_nodes),
  '#required' => TRUE,
];
```

### Processing Form Values

When the form is submitted, the values will be available in the form state as an array of entity IDs:

```php
public function submitForm(array &$form, FormStateInterface $form_state) {
  $values = $form_state->getValue('my_entities');
  
  // Process the submitted values
  foreach ($values['items'] as $key => $item) {
    if (!empty($item['target_id'])) {
      $entity_id = $item['target_id'];
      // Do something with the entity ID
      $entity = \Drupal::entityTypeManager()
        ->getStorage('node')
        ->load($entity_id);
    }
  }
}
```

### Working with Different Entity Types

#### User References
```php
$form['team_members'] = [
  '#type' => 'entity_reference_autocomplete_add_more',
  '#title' => t('Team Members'),
  '#target_type' => 'user',
  '#selection_settings' => [
    'include_anonymous' => FALSE,
  ],
];
```

#### Taxonomy Terms
```php
$form['tags'] = [
  '#type' => 'entity_reference_autocomplete_add_more',
  '#title' => t('Tags'),
  '#target_type' => 'taxonomy_term',
  '#selection_settings' => [
    'target_bundles' => ['tags'],
  ],
];
```

## Features

- **Dynamic Add/Remove**: Users can add multiple entity references and remove them individually
- **AJAX Support**: Adding and removing items works without page refresh
- **Autocomplete**: Built-in entity autocomplete functionality
- **Validation**: Supports standard Drupal form validation
- **Flexible**: Works with any entity type and bundle configuration