<?php

namespace Drupal\Tests\entity_reference_autocomplete_add_more\Unit\Element;

use Drupal\Tests\UnitTestCase;
use Drupal\entity_reference_autocomplete_add_more\Element\EntityReferenceAutocompleteAddMore;
use Drupal\Core\Form\FormStateInterface;

/**
 * @coversDefaultClass \Drupal\entity_reference_autocomplete_add_more\Element\EntityReferenceAutocompleteAddMore
 * @group entity_reference_autocomplete_add_more
 */
class EntityReferenceAutocompleteAddMoreTest extends UnitTestCase {

  /**
   * Test if getinfo function returns default array structure.
   */
  public function testGetInfo() {
    $element = (new EntityReferenceAutocompleteAddMore([], 'entity_reference_autocomplete_add_more', []))->getInfo();
    $this->assertArrayHasKey('#input', $element);
    $this->assertArrayHasKey('#process', $element);
    $this->assertArrayHasKey('#element_validate', $element);
    $this->assertEquals('Entity Reference Autocomplete Add More', $element['#title']);
  }

  /**
   * Test if remove field name is correct format.
   */
  public function testGetRemoveFieldName() {
    $result = $this->callProtectedMethod('getRemoveFieldName', ['2', 'testfield']);
    $this->assertEquals('remove_testfield_2', $result);
  }

  /**
   * Test if the selector is in correct format.
   */
  public function testGetSelector() {
    $result = $this->callProtectedMethod('getSelector', ['testfield[0]', TRUE]);
    $this->assertEquals('#testfield-0--wrapper', $result);
    $result2 = $this->callProtectedMethod('getSelector', ['testfield[0]', FALSE]);
    $this->assertEquals('testfield-0--wrapper', $result2);
  }

  /**
   * Test if container name is correct format.
   */
  public function testGetContainerName() {
    $result = $this->callProtectedMethod('getContainerName', ['testfield[0]', TRUE]);
    $this->assertEquals('#testfield-0--container', $result);
    $result2 = $this->callProtectedMethod('getContainerName', ['testfield[0]', FALSE]);
    $this->assertEquals('testfield-0--container', $result2);
  }

  /**
   * Test if item count is correct logic.
   */
  public function testGetItemCount() {
    $form_state = $this->getMockBuilder(FormStateInterface::class)->getMock();
    $form_state->expects($this->any())->method('get')->willReturn(FALSE);
    $element = ['#name' => 'testfield', '#default_value' => [1, 2, 3]];
    $result = $this->callProtectedMethod('getItemCount', [$form_state, $element]);
    $this->assertEquals(3, $result);
    $element2 = ['#name' => 'testfield', '#default_value' => []];
    $result2 = $this->callProtectedMethod('getItemCount', [$form_state, $element2]);
    $this->assertEquals(1, $result2);
  }

  /**
   * Test if element type returns the right value.
   */
  public function testGetElementType() {
    $this->assertEquals('entity_reference_autocomplete_add_more', EntityReferenceAutocompleteAddMore::getElementType());
  }

  /**
   * Check if exists, this function is required.
   */
  public function testValidateElement() {
    $element = [];
    $form_state = $this->getMockBuilder(FormStateInterface::class)->getMock();
    $complete_form = [];
    EntityReferenceAutocompleteAddMore::validateElement($element, $form_state, $complete_form);
    // No-op, just ensure callable.
    $this->assertTrue(TRUE);
  }

  /**
   * Check if process element returns the correct array.
   */
  public function testProcessElement() {
    $element = [
      '#name' => 'testfield',
      '#title' => 'TestField',
      '#target_type' => 'taxonomy_term',
      '#selection_settings' => ['target_bundles' => ['agency']],
      '#default_value' => [1, 2],
      '#parents' => ['form', 'testfield'],
    ];
    $form_state = $this->getMockBuilder(FormStateInterface::class)->getMock();
    $form_state->expects($this->any())->method('get')->willReturn(FALSE);
    $result = EntityReferenceAutocompleteAddMore::processElement($element, $form_state);
    $this->assertArrayHasKey('items', $result);
    $this->assertArrayHasKey('add', $result['items']);
  }

  /**
   * Check if ajax callback returns the correct response.
   */
  public function testAjaxCallback() {
    $form = [
      'form' => [
        'testfield' => [
          'items' => [1, 2],
          'add' => [],
          '#prefix' => '',
          '#suffix' => '',
        ],
      ],
    ];
    $form_state = $this->getMockBuilder(FormStateInterface::class)->getMock();
    $form_state->expects($this->any())->method('getTriggeringElement')->willReturn([
      '#parents' => ['form', 'testfield'],
      '#attributes' => ['data-element-name' => 'testfield'],
    ]);
    $form_state->expects($this->any())->method('get')->willReturn('testfield');
    $result = EntityReferenceAutocompleteAddMore::ajaxCallback($form, $form_state);
    $this->assertIsArray($result);
  }

  /**
   * Check if add remove items are callable and no errors.
   */
  public function testAddItemAndRemoveItem() {
    $form = [];
    $form_state = $this->getMockBuilder(FormStateInterface::class)->getMock();
    $form_state->expects($this->any())->method('getTriggeringElement')->willReturn([
      '#attributes' => ['data-element-name' => 'testfield', 'data-index' => 1],
    ]);
    $form_state->expects($this->any())->method('set');
    $form_state->expects($this->any())->method('setRebuild');
    EntityReferenceAutocompleteAddMore::addItem($form, $form_state);
    EntityReferenceAutocompleteAddMore::removeItem($form, $form_state);
    $this->assertTrue(TRUE);
  }

  /**
   * Helper to call protected static methods.
   */
  protected function callProtectedMethod($method, array $args) {
    $class = new \ReflectionClass(EntityReferenceAutocompleteAddMore::class);
    $m = $class->getMethod($method);
    $m->setAccessible(TRUE);
    return $m->invokeArgs(NULL, $args);
  }

}
