<?php

namespace Drupal\entity_reference_autocomplete_add_more\Element;

use Drupal\Core\Render\Element\FormElementBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides an entity reference autocomplete form element with add more.
 *
 * @FormElement("entity_reference_autocomplete_add_more")
 */
class EntityReferenceAutocompleteAddMore extends FormElementBase {

  protected const ELEMENT_ROOT = 'items';

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    return [
      '#input' => TRUE,
      '#title' => 'Entity Reference Autocomplete Add More',
      '#target_type' => '',
      '#default_value' => [],
      '#selection_settings' => [],
      '#element_validate' => [get_called_class() . '::validateElement'],
      '#process' => [get_called_class() . '::processElement'],
      '#theme_wrappers' => ['form_element'],
    ];
  }

  /**
   * Generate remove button field name.
   *
   * @param string $index
   *   The index of the remove button.
   * @param string $element_name
   *   The parent form field using the element.
   *
   * @return string
   *   Return the calculated field name from above.
   */
  protected static function getRemoveFieldName(string $index, string $element_name) {
    return 'remove_' . $element_name . '_' . $index;
  }

  /**
   * Generate a wrapper name safe to use as element id.
   *
   * @param string $element_name
   *   The name of the element with parents in [].
   * @param bool $css
   *   If return value should be a CSS selector.
   *
   * @return string
   *   Return a selector name safe to use as JS ID selector.
   */
  protected static function getSelector(string $element_name, $css = TRUE) {
    return ($css ? '#' : '') . preg_replace("/[^A-Za-z0-9 ]/", "-", $element_name) . '-wrapper';
  }

  /**
   * Generate a container name safe to use as element id.
   *
   * @param string $element_name
   *   The name of the element with parents in [].
   * @param bool $css
   *   If return value should be a CSS selector.
   *
   * @return string
   *   Return a selector name safe to use as JS class selector.
   */
  protected static function getContainerName(string $element_name, $css = TRUE) {
    return ($css ? '#' : '') . preg_replace("/[^A-Za-z0-9 ]/", "-", $element_name) . '-container';
  }

  /**
   * Returns current number of items added using add / remove buttons.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object to get the items from.
   * @param array $element
   *   The element array.
   *
   * @return int|void
   *   Number of items added by the element.
   */
  protected static function getItemCount(FormStateInterface $form_state, array $element) {
    $num_items = $form_state->get([$element['#name'], 'num_items']) ?: FALSE;
    if ($num_items) {
      return $num_items;
    }

    return !empty($element['#default_value']) ? count($element['#default_value']) : 1;
  }

  /**
   * Get form element type.
   *
   * @return string
   *   The name of the FormElement.
   */
  public static function getElementType() {
    return 'entity_reference_autocomplete_add_more';
  }

  /**
   * Generate an element array to add a new element.
   *
   * @param array $element
   *   The  element array.
   * @param \Drupal\Core\Entity\EntityInterface $default
   *   If there's a default entity available.
   * @param int $index
   *   The index of the element.
   *
   * @return array
   *   Generated element array
   */
  public static function getElementItem(array $element, $default, int $index = 0) {
    $element_item = [
      '#type' => 'container',
      '#attributes' => [
        'class' => $element['#name'] . '-item-container-' . $index,
      ],
      '#weight' => $index,
    ];
    $element_item['target_id'] = [
      '#type' => 'entity_autocomplete',
      '#target_type' => $element['#target_type'],
      '#selection_settings' => $element['#selection_settings'] ?: [],
      '#default_value' => $default,
      '#required' => $element['#required'] ?? FALSE,
      '#maxlength' => 1024,
      '#weight' => 0,
    ];
    // Show remove button for more than one item.
    if ($index > 0) {
      $element_item['remove'] = [
        '#type' => 'submit',
        '#name' => $element['#title'] . ' ' . $index,
        '#value' => t('Remove'),
        '#submit' => [[get_called_class(), 'removeItem']],
        '#ajax' => [
          'callback' => [get_called_class(), 'ajaxCallback'],
          'event' => 'click',
          'wrapper' => self::getSelector($element['#name'], FALSE),
        ],
        '#attributes' => [
          'data-index' => $index,
          'data-element-name' => $element['#name'],
        ],
        '#limit_validation_errors' => [],
        '#weight' => 1,
      ];
    }
    return $element_item;
  }

  /**
   * Generate an element array to add a new element.
   *
   * @param array $element
   *   The element array.
   *
   * @return array
   *   Generated element array
   */
  public static function getElementItemsWrapper(array $element) {
    $element_item = [
      '#type' => 'container',
      '#title' => $element['#title'] ? $element['#title'] : '',
      '#attributes' => ['id' => self::getContainerName($element['#name'], FALSE)],
    ];
    // Add another button.
    $element_item['add'] = [
      '#type' => 'submit',
      '#value' => t('Add another item'),
      '#submit' => [[get_called_class(), 'addItem']],
      '#ajax' => [
        'callback' => [get_called_class(), 'ajaxCallback'],
        'event' => 'click',
        'wrapper' => self::getSelector($element['#name'], FALSE),
      ],
      '#attributes' => [
        'data-element-name' => $element['#name'],
      ],
      '#limit_validation_errors' => [],
      '#weight' => 1000000,
    ];

    return $element_item;
  }

  /**
   * Get current items from form state or default values.
   *
   * @param mixed $form_state
   *   The form state object to get the items from.
   * @param mixed $element_name
   *   The name of the element with parents in [].
   */
  public static function getCurrentItems($form_state, $element_name) {
    $current_items = $form_state->get([$element_name, 'current_items']);
    $default_values = $form_state->get([$element_name, 'default_values']);
    if ($current_items) {
      return $current_items;
    }
    elseif (!empty($default_values)) {
      foreach ($default_values as $key => $value) {
        $current_items[] = $value;
      }
    }

    return $current_items;
  }

  /**
   * Generate the form element based on either default value or current state.
   *
   * @param array $element
   *   The element array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return mixed
   *   The processed element array.
   */
  public static function processElement(array &$element, FormStateInterface $form_state) {
    $element_type = self::getElementType();
    $last_element = end($element['#parents']);
    $form_state->set([$element_type, $last_element], $element['#parents']);
    // If it's an edit form where previous values exist.
    $default_values = !empty($element['#default_value']) ? $element['#default_value'] : NULL;
    $form_state->set([$element['#name'], 'default_values'], $default_values);
    $current_items = self::getCurrentItems($form_state, $element['#name']);
    $form_state->set([$element['#name'], 'element_parent'], $last_element);
    if (!$current_items) {
      $form_state->set([$element['#name'], 'current_items'], [NULL]);
    }

    $element = $element + [
      '#prefix' => '<div id="' . self::getSelector($element['#name'], FALSE) . '">',
      '#suffix' => '</div>',
    ];
    $element[self::ELEMENT_ROOT] = self::getElementItemsWrapper($element);
    if ($current_items) {
      // Create the form based on default or current status.
      foreach ($current_items as $i => $entity) {
        if ($entity !== 'removed') {
          $element[self::ELEMENT_ROOT][$i] = self::getElementItem($element, $entity, $i);
        }
      }
    }
    else {
      $element[self::ELEMENT_ROOT][0] = self::getElementItem($element, NULL, 0);
    }

    return $element;
  }

  /**
   * Return updated element to re-render.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return array|mixed
   *   The generated element object to replace original element.
   */
  public static function ajaxCallback(array &$form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement();
    $parents = $trigger['#parents'];
    $last_element = $form_state->get([$trigger['#attributes']['data-element-name'], 'element_parent']);
    if (count($trigger['#parents']) > 1) {
      foreach ($parents as $key) {
        $form = $form[$key];
        if ($key == $last_element) {
          break;
        }
      }
    }
    else {
      $form = $form[$last_element];
    }

    return $form;
  }

  /**
   * Handler for add button, sets add flag.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public static function addItem(array &$form, FormStateInterface $form_state) {
    // Retrieve the element name stored in form state to set a flag.
    $element_name = $form_state->getTriggeringElement()['#attributes']['data-element-name'];
    $current_items = self::getCurrentItems($form_state, $element_name);
    $current_items[] = NULL;
    $form_state->set([$element_name, 'current_items'], $current_items);
    $form_state->setRebuild();
  }

  /**
   * Get the index of the remove button to remove exact item.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public static function removeItem(array &$form, FormStateInterface $form_state) {
    $index = $form_state->getTriggeringElement()['#attributes']['data-index'];
    $element_name = $form_state->getTriggeringElement()['#attributes']['data-element-name'];
    $current_items = self::getCurrentItems($form_state, $element_name);
    $current_items[$index] = 'removed';
    $form_state->set([$element_name, 'current_items'], $current_items);
    $form_state->setRebuild();
  }

  /**
   * Implement any validation logic on this element.
   *
   * @param array $element
   *   The element array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   * @param array $complete_form
   *   The full form array.
   */
  public static function validateElement(&$element, FormStateInterface $form_state, &$complete_form) {
    // No-op, but you can add validation here if needed.
  }

}
