<?php

namespace Drupal\entity_reference_direct_input\EntityReferenceDirectInput;

use Drupal\Core\Path\PathValidatorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityAutocompleteMatcher;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Enhanced entity autocomplete matcher for major core entity types.
 *
 * Extends the core EntityAutocompleteMatcher to allow direct input of:
 * - Entity IDs (e.g., "123" or "#123")
 * - Email addresses (for users only, e.g., "user@example.com")
 * - Full URLs to entities
 * - Path aliases.
 *
 * Supported entity types:
 * - Node
 * - User (with email support)
 * - Taxonomy Term
 *
 * This provides immediate autocomplete suggestions without requiring
 * users to search through the autocomplete dropdown.
 */
class EntityReferenceDirectInputMatcher extends EntityAutocompleteMatcher {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The path validator.
   *
   * @var \Drupal\Core\Path\PathValidatorInterface
   */
  protected $pathValidator;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity reference selection handler plugin manager.
   *
   * @var \Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManagerInterface
   */
  protected $selectionManager;

  /**
   * Sets the config factory.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function setConfigFactory(ConfigFactoryInterface $config_factory): void {
    $this->configFactory = $config_factory;
  }

  /**
   * Sets the module handler.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function setModuleHandler(ModuleHandlerInterface $module_handler): void {
    $this->moduleHandler = $module_handler;
  }

  /**
   * Sets the request stack.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function setRequestStack(RequestStack $request_stack): void {
    $this->requestStack = $request_stack;
  }

  /**
   * Sets the path validator.
   *
   * @param \Drupal\Core\Path\PathValidatorInterface $path_validator
   *   The path validator.
   */
  public function setPathValidator(PathValidatorInterface $path_validator): void {
    $this->pathValidator = $path_validator;
  }

  /**
   * Sets the entity type manager.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function setEntityTypeManager(EntityTypeManagerInterface $entity_type_manager): void {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Gets matches for entity autocomplete suggestions.
   *
   * Adds a direct match when the user types/pastes URL/alias/#ID/ID.
   * Supports all core entity types with per-entity-type configuration.
   *
   * @param string $target_type
   *   The target entity type.
   * @param string $selection_handler
   *   The selection handler.
   * @param array $selection_settings
   *   The selection settings.
   * @param string $string
   *   The string to match.
   *
   * @return array
   *   An array of matches.
   */
  public function getMatches($target_type, $selection_handler, $selection_settings, $string = '') {
    // First get core's matches.
    $matches = parent::getMatches($target_type, $selection_handler, $selection_settings, $string);

    // Check if this entity type is enabled in configuration.
    $config = $this->configFactory->get('entity_reference_direct_input.settings');
    $enabled_entities = $config->get('enabled_entities') ?? [];

    if (!in_array($target_type, $enabled_entities, TRUE) || $string === '') {
      return $matches;
    }

    $raw = trim((string) $string);
    $entity_id = $this->resolveToEntityId($raw, $target_type);
    if (!$entity_id) {
      return $matches;
    }

    // Load the entity.
    $entity = $this->entityTypeManager->getStorage($target_type)->load($entity_id);
    if (!$entity) {
      return $matches;
    }

    // Enforce target bundle restrictions if provided.
    $allowed_bundles = $selection_settings['target_bundles'] ?? NULL;
    if (is_array($allowed_bundles) && $allowed_bundles && !in_array($entity->bundle(), $allowed_bundles, TRUE)) {
      // Disallowed by field handler; don't inject.
      return $matches;
    }

    // If not already present, prepend our direct match.
    $exists = array_filter($matches, function ($m) use ($entity_id) {
      return preg_match('/\((\d+)\)\s*$/', $m['value'], $mm) && (int) $mm[1] === $entity_id;
    });

    if (!$exists) {
      $label = $this->getEntityLabel($entity, $target_type);
      $alias = $this->getEntityAlias($entity, $target_type);

      array_unshift($matches, [
        // What widget expects.
        'value' => sprintf('%s (%d)', $label, $entity_id),
        // What user sees.
        'label' => sprintf('%s — %s (id: %d)', $label, $alias, $entity_id),
      ]);
    }

    return $matches;
  }

  /**
   * Resolve input like "123", "#123", email, full URL, or alias to entity ID.
   *
   * @param string $input
   *   The input string to resolve.
   * @param string $entity_type
   *   The entity type to resolve for.
   *
   * @return int|null
   *   The entity ID if found, NULL otherwise.
   */
  private function resolveToEntityId(string $input, string $entity_type): ?int {
    // 1) #123 or 123
    if (preg_match('/^\#?(\d+)$/', $input, $m)) {
      return (int) $m[1];
    }

    // 2) Email address (for users only)
    if ($entity_type === 'user' && filter_var($input, FILTER_VALIDATE_EMAIL)) {
      $user_storage = $this->entityTypeManager->getStorage('user');
      $users = $user_storage->loadByProperties(['mail' => $input]);
      if (!empty($users)) {
        $user = reset($users);
        return (int) $user->id();
      }
    }

    // 3) Full URL → path
    if (preg_match('@^https?://@i', $input)) {
      $input = (string) (parse_url($input, PHP_URL_PATH) ?: $input);
    }

    // Handle paths that don't start with /.
    if (!str_starts_with($input, '/')) {
      $input = '/' . $input;
    }

    // Strip base path (subdir installs) and decode.
    $base = $this->requestStack->getCurrentRequest()->getBasePath();
    if ($base && str_starts_with($input, $base . '/')) {
      $input = substr($input, strlen($base));
    }
    $path = '/' . ltrim(urldecode($input), '/');

    // 4) Alias/path → routed URL → entity ID (language-safe).
    $url = $this->pathValidator->getUrlIfValidWithoutAccessCheck($path);
    if ($url && $url->isRouted()) {
      $route_name = $url->getRouteName();
      $route_parameters = $url->getRouteParameters();

      // Map route names to entity types and parameter names.
      $route_mapping = [
        'entity.node.canonical' => ['entity_type' => 'node', 'param' => 'node'],
        'entity.user.canonical' => ['entity_type' => 'user', 'param' => 'user'],
        'entity.taxonomy_term.canonical' => ['entity_type' => 'taxonomy_term', 'param' => 'taxonomy_term'],
      ];

      if (isset($route_mapping[$route_name])) {
        $mapping = $route_mapping[$route_name];
        if ($mapping['entity_type'] === $entity_type && isset($route_parameters[$mapping['param']])) {
          return (int) $route_parameters[$mapping['param']];
        }
      }
    }

    // 5) Try direct path patterns as fallback.
    if (preg_match('#^/node/(\d+)$#', $path, $matches) && $entity_type === 'node') {
      return (int) $matches[1];
    }
    if (preg_match('#^/user/(\d+)$#', $path, $matches) && $entity_type === 'user') {
      return (int) $matches[1];
    }
    if (preg_match('#^/taxonomy/term/(\d+)$#', $path, $matches) && $entity_type === 'taxonomy_term') {
      return (int) $matches[1];
    }

    return NULL;
  }

  /**
   * Get the display label for an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param string $entity_type
   *   The entity type.
   *
   * @return string
   *   The entity label.
   */
  private function getEntityLabel($entity, string $entity_type): string {
    // Most entities have a label() method.
    if (method_exists($entity, 'label')) {
      return $entity->label() ?: 'Untitled';
    }

    // Fallback for entities without label method.
    return 'Untitled';
  }

  /**
   * Get the alias/path for an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param string $entity_type
   *   The entity type.
   *
   * @return string
   *   The entity alias or path.
   */
  private function getEntityAlias($entity, string $entity_type): string {
    // Get the canonical URL for the entity.
    $url = $entity->toUrl('canonical');
    $path = $url->toString();

    // Try to get alias for nodes and taxonomy terms if path_alias is available.
    if (in_array($entity_type, ['node', 'taxonomy_term']) && $this->moduleHandler->moduleExists('path_alias')) {
      $alias_storage = $this->entityTypeManager->getStorage('path_alias');
      $alias_entities = $alias_storage->loadByProperties(['path' => $path]);
      if (!empty($alias_entities)) {
        /** @var \Drupal\path_alias\Entity\PathAlias $alias_entity */
        $alias_entity = reset($alias_entities);
        return $alias_entity->get('alias')->value;
      }
    }

    return $path;
  }

}
