<?php

declare(strict_types=1);

namespace Drupal\entity_reference_field_autocomplete_filter\Plugin\Field\FieldWidget;

use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldWidget\EntityReferenceAutocompleteWidget;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use function count;

/**
 * Plugin for the 'entity_reference_filterable_autocomplete' widget.
 *
 * @FieldWidget(
 *   id = "entity_reference_filterable_autocomplete",
 *   label = @Translation("Filterable Autocomplete"),
 *   description = @Translation("A filterable autocomplete text field."),
 *   field_types = {
 *     "entity_reference"
 *   }
 * )
 */
class EntityReferenceFilterableAutocompleteWidget extends EntityReferenceAutocompleteWidget {

  /**
   * Constructs a new instance of the object.
   *
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $settings
   *   An array of settings.
   * @param array $third_party_settings
   *   An array of third party settings.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $bundleInfo
   *   The bundle info service.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    private readonly EntityTypeBundleInfoInterface $bundleInfo,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('entity_type.bundle.info'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $fieldName = $this->fieldDefinition->getName();

    /** @var \Drupal\Core\Field\EntityReferenceFieldItemListInterface $items */
    $referencedEntities = $items->referencedEntities();
    $thisEntity = $referencedEntities[$delta] ?? NULL;
    $thisEntityBundle = $thisEntity?->bundle();

    // Get the bundle selection from user input using proper nested array access.
    $selectedBundle = $this->getSelectedBundleFromFormState($form_state, $element, $fieldName, $delta, $thisEntityBundle);
    $enabledBundles = $this->getFieldSetting('handler_settings')['target_bundles'];
    $bundleOptions = $this->bundleOptions($enabledBundles);

    // If there is one or fewer bundles, leave the default widget.
    if (count($bundleOptions) <= 1) {
      return $element;
    }

    if ($selectedBundle) {
      $element['target_id']['#selection_settings']['target_bundles'] = [$selectedBundle];
    }
    else {
      // Limit to the bundles defined in the field settings.
      $element['target_id']['#selection_settings']['target_bundles'] = $enabledBundles;
    }

    // Add the bundle select field with AJAX callback.
    $wrapperId = Html::getClass($fieldName) . '-target-id-wrapper-' . $delta;
    $element['bundle'] = [
      '#type' => 'select',
      '#title' => $this->t('Search within'),
      '#options' => $bundleOptions,
      '#weight' => -1,
      '#empty_option' => $this->t('- Any bundle -'),
      '#default_value' => $thisEntityBundle,
      '#ajax' => [
        'callback' => [static::class, 'updateTargetBundles'],
        'wrapper' => $wrapperId,
      ],
      '#wrapper_attributes' => [
        'class' => ['entity-reference-filterable-autocomplete-bundle'],
      ],
    ];

    // Wrap the target_id field so that it can be updated via AJAX.
    // Also wrap both the target_id & bundle fields for styling purposes.
    $element['bundle']['#prefix'] = '<div class="entity-reference-filterable-autocomplete-wrapper">';
    $element['target_id']['#prefix'] = '<div id="' . $wrapperId . '">';
    $element['target_id']['#suffix'] = '</div></div>';

    $element['#attached']['library'][] = 'entity_reference_field_autocomplete_filter/theme';

    return $element;
  }

  /**
   * Gets the selected bundle from form state using proper nested array access.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   * @param array $element
   *   The form element array.
   * @param string $fieldName
   *   The field name.
   * @param int $delta
   *   The field delta.
   * @param string|null $fallback
   *   The fallback bundle value.
   *
   * @return string|null
   *   The selected bundle or the fallback value.
   */
  private function getSelectedBundleFromFormState(FormStateInterface $form_state, array $element, string $fieldName, int $delta, ?string $fallback): ?string {
    // Build the parents array for this field element.
    $parents = $element['target_id']['#field_parents'];
    $field_info = [$fieldName, $delta, 'bundle'];
    $parents = array_merge($parents, $field_info);

    // Get the user input using nested array access.
    $input = $form_state->getUserInput();
    $selectedBundle = NestedArray::getValue($input, $parents);

    return $selectedBundle ?? $fallback;
  }

  /**
   * Retrieves an array of bundle options with their labels.
   *
   * @param array|null $enabledBundles
   *   An array of enabled bundle IDs, or null to allow all bundles.
   *
   * @return array
   *   An associative array of bundle IDs => labels.
   */
  private function bundleOptions(?array $enabledBundles): array {
    $fieldTargetType = $this->getFieldSetting('target_type');
    $bundleInfo = $this->bundleInfo->getBundleInfo($fieldTargetType);

    // When no bundles are selected then all are valid options.
    if ($enabledBundles === NULL) {
      $enabledBundles = array_keys($bundleInfo);
    }

    $options = [];
    foreach ($enabledBundles as $bundle) {
      // If a label exists use that, else fall back to the machine name.
      $label = $bundleInfo[$bundle]['label'] ?? $bundle;
      $options[$bundle] = $label;
    }

    return $options;
  }

  /**
   * Respond to update target bundles ajax request.
   *
   * @param array $form
   *   The form structure from which the target bundles are retrieved.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form state object.
   *
   * @return array
   *   The updated target_id field.
   */
  public static function updateTargetBundles(array $form, FormStateInterface $formState): array {
    $triggeringElement = $formState->getTriggeringElement();

    // Get the parents of the triggering element and navigate to the target_id field.
    $parents = $triggeringElement['#array_parents'];

    // Replace 'bundle' with 'target_id' in the parents array.
    $parents[count($parents) - 1] = 'target_id';

    // Use NestedArray to get the target_id element from the nested form structure.
    return NestedArray::getValue($form, $parents);
  }

}
