<?php

namespace Drupal\Tests\entity_reference_field_autocomplete_filter\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\user\UserInterface;

/**
 * Tests the field widget.
 *
 * @group entity_reference_field_autocomplete_filter
 */
class EntityReferenceFilterableAutocompleteWidgetTest extends WebDriverTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with permission to manage and use the field.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $adminUser;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'field_ui',
    'entity_reference_field_autocomplete_filter',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->drupalCreateContentType([
      'type' => 'page',
      'name' => 'Basic page',
      'display_submitted' => FALSE,
    ]);
    $this->drupalCreateContentType([
      'type' => 'article',
      'name' => 'Article',
      'display_submitted' => FALSE,
    ]);
    $this->drupalCreateContentType([
      'type' => 'event',
      'name' => 'Event',
      'display_submitted' => FALSE,
    ]);
    $this->drupalCreateNode([
      'type' => 'event',
      'title' => 'Event 1',
    ]);
    $this->drupalCreateNode([
      'type' => 'event',
      'title' => 'Shared from event',
    ]);
    $this->drupalCreateNode([
      'type' => 'article',
      'title' => 'Article 1',
    ]);
    $this->drupalCreateNode([
      'type' => 'article',
      'title' => 'Shared from article',
    ]);
    $this->drupalCreateNode([
      'type' => 'page',
      'title' => 'Page 1',
    ]);
    $this->drupalCreateNode([
      'type' => 'page',
      'title' => 'Shared from page',
    ]);
    $this->adminUser = $this->drupalCreateUser([
      'access content',
      'bypass node access',
      'administer nodes',
      'administer content types',
      'administer node fields',
      'administer node form display',
    ]);
    $this->drupalLogin($this->adminUser);
  }

  /**
   * Tests the widget.
   */
  public function testWidget() {
    // Create the reference field itself.
    $this->drupalGet('admin/structure/types/manage/page/fields/add-field/reference/true', ['query' => ['entity_type' => 'node']]);

    $this->submitForm([
      'field_options_wrapper' => 'field_ui:entity_reference:node',
      'label' => 'Reference',
    ], 'Continue');

    // Select limited target bundles, leaving out 'page' here.
    $this->getSession()->getPage()->checkField('settings[handler_settings][target_bundles][article]');
    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->getSession()->getPage()->checkField('settings[handler_settings][target_bundles][event]');
    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->submitForm([], 'Save');

    $this->getSession()->wait(2000);
    $this->assertSession()->pageTextContains("Saved Reference configuration");

    // Change form display to this widget.
    $this->drupalGet('admin/structure/types/manage/page/form-display');
    $this->getSession()->getPage()->selectFieldOption('fields[field_reference][type]', 'entity_reference_filterable_autocomplete');
    $this->submitForm([
    ], 'Save');

    // Use the widget now.
    $this->drupalGet('node/add/page');
    $this->assertSession()->fieldExists('field_reference[0][bundle]');

    // Check that with article and event allowed, only the 2 shared items are
    // suggested.
    $this->getSession()->getPage()->fillField('field_reference[0][target_id]', 'Shared');
    $this->assertSession()->waitOnAutocomplete();
    $results = $this->getSession()->getPage()->findAll('css', '.ui-autocomplete li');
    $texts = [];
    foreach ($results as $result) {
      $texts[] = $result->getText();
    }
    sort($texts);
    $this->assertSame([
      'Shared from article',
      'Shared from event',
    ], $texts);

    // Now restrict to article only.
    $this->getSession()->getPage()->selectFieldOption('field_reference[0][bundle]', 'article');
    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->getSession()->getPage()->fillField('field_reference[0][target_id]', 'Shared');
    $this->assertSession()->waitOnAutocomplete();
    $results = $this->getSession()->getPage()->findAll('css', '.ui-autocomplete li');
    $texts = [];
    foreach ($results as $result) {
      $texts[] = $result->getText();
    }
    sort($texts);
    $this->assertSame([
      'Shared from article',
    ], $texts);

    // Change to a single bundle.
    $this->drupalGet('admin/structure/types/manage/page/fields/node.page.field_reference');
    $this->getSession()->getPage()->uncheckField('settings[handler_settings][target_bundles][article]');
    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->submitForm([
      'settings[handler_settings][target_bundles][article]' => FALSE,
    ], 'Save settings');
    $this->assertSession()->waitForElementRemoved('css', '.ajax-progress');
    $this->drupalGet('admin/structure/types/manage/page/fields/node.page.field_reference');
    $this->drupalGet('node/add/page');
    $this->assertSession()->fieldNotExists('field_reference[0][bundle]');
  }

}
