<?php

namespace Drupal\entity_reference_integrity_enforce;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\entity_reference_integrity_enforce\Exception\ProtectedEntityException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Hook into entity deletes and throw an exception to prevent them disappearing.
 */
class EntityPredelete implements ContainerInjectionInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity type IDs protection is enabled for.
   *
   * @var array
   */
  protected $enabledEntityTypeIds;

  /**
   * Create a EntityPredelete object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param array $enabled_entity_type_ids
   *   The entity type IDs protection is enabled for.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, array $enabled_entity_type_ids) {
    $this->entityTypeManager = $entity_type_manager;
    $this->enabledEntityTypeIds = $enabled_entity_type_ids;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('config.factory')->get('entity_reference_integrity_enforce.settings')->get('enabled_entity_type_ids')
    );
  }

  /**
   * Implements hook_entity_delete().
   */
  public function entityDelete(EntityInterface $entity) {
    if (!in_array($entity->getEntityTypeId(), $this->enabledEntityTypeIds, TRUE)) {
      return;
    }
    $handler = $this->entityTypeManager->getHandler($entity->getEntityTypeId(), 'entity_reference_integrity');
    if ($handler->hasDependents($entity)) {
      throw new ProtectedEntityException(sprintf('Cannot delete "%s" of type "%s" with label "%s" and ID "%s" because other content is referencing it and the integrity of this entity type is enforced.', $entity->getEntityTypeId(), $entity->bundle(), $entity->label(), $entity->id()));
    }
  }

}
