<?php

namespace Drupal\entity_reference_integrity_enforce;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Alter entity delete forms to provide some warning deletes will fail.
 */
class FormAlter implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Create a FormAlter object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory, RendererInterface $renderer) {
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('config.factory'),
      $container->get('renderer'),
    );
  }

  /**
   * Implements hook_form_alter().
   */
  public function formAlter(&$form, FormStateInterface $form_state, $form_id) {
    /** @var \Drupal\Core\Entity\EntityFormInterface $form_object */
    $form_object = $form_state->getFormObject();
    if (!$this->isDeleteForm($form_object)) {
      return;
    }

    // Load the module configuration.
    $config = $this->configFactory->get('entity_reference_integrity_enforce.settings');

    // Ensure that the render cache is cleared for this form when the
    // entity_reference_integrity configuration changes.
    $this->renderer->addCacheableDependency($form, $config);

    $entity = $form_object->getEntity();

    /** @var \Drupal\entity_reference_integrity\EntityReferenceIntegrityEntityHandlerInterface $handler */
    $handler = $this->entityTypeManager->getHandler($entity->getEntityTypeId(), 'entity_reference_integrity');

    if (in_array($entity->getEntityTypeId(), $config->get('enabled_entity_type_ids'), TRUE) && $handler->hasDependents($entity)) {
      $referencing_entity_ids = $handler->getDependentEntityIds($entity);

      unset($form['actions']['submit']);
      unset($form['description']);

      $form['referencing_entities_list'] = [
        '#weight' => -10,
        'explanation' => [
          '#prefix' => '<div  class="messages messages--error">',
          '#markup' => $this->t('You can not delete this as it is being referenced by another entity.'),
          '#suffix' => '</div>',
        ],
        'entities' => $this->buildReferencingEntitiesList($referencing_entity_ids),
        '#suffix' => '<br/>',
      ];
    }
  }

  /**
   * Build a UI for listing the referencing entities.
   *
   * @param array $referencing_entity_ids
   *   An array of referencing entity IDs, organized into sub-arrays by entity
   *   type.
   *
   * @return array
   *   A renderable array of referencing entities.
   */
  protected function buildReferencingEntitiesList(array $referencing_entity_ids) {
    $build = [];
    // Load the first 11 entities of each type. We only display 10 on this page
    // to prevent OOM errors, so 11 will tell us whether we need to display a
    // message.
    $referencing_entities = [];
    foreach ($referencing_entity_ids as $entity_type_id => $entity_ids) {
      $referencing_entities[$entity_type_id] = $this->entityTypeManager->getStorage($entity_type_id)->loadMultiple(array_slice($entity_ids, 0, 11));
    }
    foreach ($referencing_entities as $entity_type_id => $entities) {
      /** @var \Drupal\Core\Entity\EntityInterface[] $entities */
      $build[$entity_type_id]['label'] = [
        '#type' => 'html_tag',
        '#tag' => 'strong',
        '#value' => reset($entities)->getEntityType()->getLabel(),
      ];
      $build[$entity_type_id]['list'] = [
        '#theme' => 'item_list',
        '#items' => [],
      ];
      $restricted_entity_count = 0;
      foreach ($entities as $entity) {
        // If there are more than 10 entities, add a message at the end.
        if (count($build[$entity_type_id]['list']['#items']) == 10 && count($entities) > 10) {
          $build[$entity_type_id]['list']['#items'][] = $this->t('... only the first 10 results are displayed.');
          break;
        }
        if ($entity->access('view label')) {
          $build[$entity_type_id]['list']['#items'][] = $entity->hasLinkTemplate('canonical') ? $entity->toLink() : $entity->label();
        }
        else {
          $restricted_entity_count++;
        }
      }
      if ($restricted_entity_count > 0) {
        $build[$entity_type_id]['list']['#items'][] = $this->t('Access to some entities is restricted.');
      }
    }
    return $build;
  }

  /**
   * Check if a given generic form is applicable to be altered.
   *
   * @param mixed $form_object
   *   The form object.
   *
   * @return bool
   *   If alteration applies.
   */
  protected function isDeleteForm($form_object) {
    return $form_object instanceof EntityFormInterface && $form_object->getOperation() === 'delete';
  }

}
