<?php

namespace Drupal\Tests\entity_reference_integrity\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\user\Entity\User;

/**
 * Test the EntityReferenceIntegrityEntityHandler.
 *
 * @coversDefaultClass \Drupal\entity_reference_integrity\EntityReferenceIntegrityEntityHandler
 *
 * @group entity_reference_integrity
 */
class EntityReferenceIntegrityEntityHandlerTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'system',
    'node',
    'user',
    'entity_reference_integrity',
  ];

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * A test user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser;

  /**
   * A test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $testNode;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installSchema('node', 'node_access');

    $this->entityTypeManager = $this->container->get('entity_type.manager');

    $this->testUser = User::create([
      'name' => 'Gerald',
    ]);
    $this->testUser->save();

    $this->testNode = Node::create([
      'title' => 'foo',
      'type' => 'bar',
    ]);
    $this->testNode->save();
  }

  /**
   * Test the test user as the author of the test node.
   */
  protected function setUserAsNodeAuthor() {
    $this->testNode->uid = $this->testUser->id();
    $this->testNode->save();
  }

  /**
   * Test the reference check method.
   *
   * @covers ::hasDependents
   */
  public function testHasDependents() {
    $handler = $this->entityTypeManager->getHandler('user', 'entity_reference_integrity');
    $this->assertFalse($handler->hasDependents($this->testUser));
    $this->setUserAsNodeAuthor();
    $this->assertTrue($handler->hasDependents($this->testUser));
  }

  /**
   * Test the reference list method.
   *
   * @covers ::getDependentEntityIds
   */
  public function testGetDependentEntityIds() {
    $handler = $this->entityTypeManager->getHandler('user', 'entity_reference_integrity');
    $this->assertEquals([], $handler->getDependentEntityIds($this->testUser));
    $this->setUserAsNodeAuthor();
    $this->assertEquals($this->testNode->id(), $handler->getDependentEntityIds($this->testUser)['node'][0]);
  }

  /**
   * Test the reference list method.
   *
   * @covers ::getDependentEntities
   */
  public function testGetDependentEntities() {
    $handler = $this->entityTypeManager->getHandler($this->testUser->getEntityTypeId(), 'entity_reference_integrity');
    $this->assertEquals([], $handler->getDependentEntities($this->testUser));
    $this->setUserAsNodeAuthor();
    $this->assertEquals($this->testNode->id(), $handler->getDependentEntities($this->testUser)['node'][0]->id());
  }

}
