<?php

namespace Drupal\entity_reference_manager\Form;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Drupal\entity_reference_manager\Service\ContentMergeManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Content Merge Form.
 *
 * Allows merging references from source entities into a target entity.
 */
class ContentMergeForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The content merge manager service.
   *
   * @var \Drupal\entity_reference_manager\Service\ContentMergeManager
   */
  protected ContentMergeManager $mergeManager;

  /**
   * The tempstore factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected PrivateTempStoreFactory $tempStoreFactory;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The path current service.
   *
   * @var \Drupal\Core\Path\CurrentPathStack
   */
  protected $pathCurrent;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected EntityTypeBundleInfoInterface $bundleInfo;

  /**
   * Constructs a ContentMergeForm object.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    ContentMergeManager $mergeManager,
    ?PrivateTempStoreFactory $tempStoreFactory = NULL,
    $routeMatch = NULL,
    $pathCurrent = NULL,
    ?EntityTypeBundleInfoInterface $bundleInfo = NULL,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->mergeManager = $mergeManager;
    $this->tempStoreFactory = $tempStoreFactory;
    $this->routeMatch = $routeMatch;
    $this->pathCurrent = $pathCurrent;
    $this->bundleInfo = $bundleInfo;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_reference_manager.manager'),
      $container->get('tempstore.private'),
      $container->get('current_route_match'),
      $container->get('path.current'),
      $container->get('entity_type.bundle.info'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'entity_reference_manager_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // Use injected route match service.
    $route_match = $this->routeMatch;
    $parameters = $route_match->getParameters()->all();
    $auto_entity_type = NULL;
    $auto_source_id = NULL;
    $auto_bundle = NULL;
    foreach (['node', 'taxonomy_term', 'media'] as $possible_type) {
      if (isset($parameters[$possible_type]) && is_object($parameters[$possible_type]) && method_exists($parameters[$possible_type], 'id')) {
        $auto_entity_type = $possible_type;
        $auto_source_id = $parameters[$possible_type]->id();
        if (method_exists($parameters[$possible_type], 'bundle')) {
          $auto_bundle = $parameters[$possible_type]->bundle();
        }
        break;
      }
    }

    // Intro message for user experience and guidance.
    $form['intro'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--info">' .
      $this->t(
          '<b>Content Merge</b> allows you to replace all references from one or more source entities into a target entity, across all reference fields in the site.<br>
          <ul>
            <li>Select the entity type, the source entities, and the target entity.</li>
            <li>When you proceed, the system will analyze all reference fields pointing to the source entities and show a summary before any changes are made.</li>
            <li>On the confirmation step, you will see all fields and the number of references that will be updated.</li>
            <li>When confirmed, all references will be updated to point to the target entity, and the source entities may be removed (unless you check "Keep source entity after merge").</li>
            <li>The process runs in batch mode for performance and safety.</li>
          </ul>
          <b>Warning:</b> This action is irreversible. It is recommended to make a backup before running the merge.'
      ) .
      '</div>',
    ];

    // Build the form with prefilled values.
    $entity_types = [
      'taxonomy_term' => $this->t('Taxonomy term'),
      'node' => $this->t('Node'),
      'media' => $this->t('Media'),
    ];

    // Always use getUserInput() for AJAX to get the latest value.
    $input = $form_state->getUserInput();
    $entity_type = $input['entity_type'] ?? $form_state->getValue('entity_type') ?? $auto_entity_type;

    // Detect entity_type change and reset bundle if needed.
    $previous_entity_type = $form_state->get('previous_entity_type');
    if ($previous_entity_type !== NULL && $previous_entity_type !== $entity_type) {
      // Limpa o valor do bundle se o entity_type mudou.
      $form_state->setValue('bundle', NULL);
      if (isset($input['bundle'])) {
        unset($input['bundle']);
      }
    }
    $form_state->set('previous_entity_type', $entity_type);

    // Prepare bundle options if entity type is selected.
    $bundle_options = [];
    // $bundle_options[] = '';
    if (!empty($entity_type)) {
      $bundle_info = $this->bundleInfo->getBundleInfo($entity_type);
      foreach ($bundle_info as $bundle_id => $bundle) {
        $bundle_options[$bundle_id] = $bundle['label'];
      }
    }

    // Detecta se a rota já fornece automaticamente o entity type.
    $is_link_task_route = !empty($auto_entity_type);

    // ENTITY TYPE.
    if ($is_link_task_route) {
      // Campo oculto, mas valor enviado normalmente.
      $form['entity_type'] = [
        '#type' => 'hidden',
        '#value' => $auto_entity_type,
      ];
    }
    else {
      // Campo visível.
      $form['entity_type'] = [
        '#type' => 'select',
        '#title' => $this->t('Entity type'),
        '#options' => $entity_types,
        '#required' => TRUE,
        '#ajax' => [
          'callback' => '::ajaxRebuild',
          'wrapper' => 'content-merge-form-wrapper',
          'event' => 'change',
        ],
        '#default_value' => $entity_type,
      ];
    }

    // Use a wrapper for AJAX rebuilds.
    $form['wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'content-merge-form-wrapper'],
    ];

    // Only show bundle field after entity_type is selected,
    // and put it inside the wrapper.
    if (!empty($entity_type) && !empty($bundle_options)) {
      $bundle_value = $input['bundle'] ?? $form_state->getValue('bundle') ?? $auto_bundle;
      if ($is_link_task_route) {
        // Bundle oculto e enviado automaticamente.
        $form['wrapper']['bundle'] = [
          '#type' => 'hidden',
          '#value' => $auto_bundle,
        ];
      }
      else {
        // Bundle visível quando não é task.
        $form['wrapper']['bundle'] = [
          '#type' => 'select',
          '#title' => $this->t('Bundle'),
          '#options' => $bundle_options,
          '#required' => TRUE,
          '#default_value' => $bundle_value,
          '#ajax' => [
            'callback' => '::ajaxRebuild',
            'wrapper' => 'content-merge-form-wrapper',
            'event' => 'change',
          ],
        ];
      }
    }

    // Get selected bundle (required for autocomplete restriction)
    $selected_bundle = $input['bundle'] ?? $form_state->getValue('bundle') ?? $auto_bundle;

    // Use bundle restriction in autocomplete if selected.
    if ($entity_type && $selected_bundle) {
      $selection_settings = [
        'target_bundles' => [$selected_bundle],
      ];
      $form['wrapper']['source'] = [
        '#type' => 'entity_autocomplete',
        '#title' => $this->t('Source entities (will be merged INTO target)'),
        '#target_type' => $entity_type,
        '#selection_settings' => $selection_settings,
        '#required' => TRUE,
        '#tags' => TRUE,
        '#default_value' => $form_state->getValue('source') ?: (
          $auto_source_id
            ? [$this->entityTypeManager->getStorage($entity_type)->load($auto_source_id)]
            : NULL
        ),
      ];
      $form['wrapper']['target'] = [
        '#type' => 'entity_autocomplete',
        '#title' => $this->t('Target entity'),
        '#target_type' => $entity_type,
        '#selection_settings' => $selection_settings,
        '#required' => TRUE,
      ];
    }
    elseif ($entity_type) {
      // Fallback if no bundle selected.
      $form['wrapper']['source'] = [
        '#type' => 'entity_autocomplete',
        '#title' => $this->t('Source entities (will be merged INTO target)'),
        '#target_type' => $entity_type,
        '#required' => TRUE,
        '#tags' => TRUE,
        '#default_value' => $form_state->getValue('source') ?: (
          $auto_source_id
            ? [$this->entityTypeManager->getStorage($entity_type)->load($auto_source_id)]
            : NULL
        ),
      ];
      $form['wrapper']['target'] = [
        '#type' => 'entity_autocomplete',
        '#title' => $this->t('Target entity'),
        '#target_type' => $entity_type,
        '#required' => TRUE,
      ];
    }

    $form['wrapper']['options'] = [
      '#type' => 'details',
      '#title' => $this->t('Options'),
      '#open' => TRUE,
    ];

    $form['wrapper']['options']['keep_source'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Keep source entity after merge'),
      '#default_value' => FALSE,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Run merge'),
      '#button_type' => 'primary',
    ];

    // Check for confirmation step.
    $confirm_data = $form_state->get('confirm_data');
    if ($confirm_data) {
      $form['confirm_message'] = [
        '#markup' => '<p>' . $this->t('The following fields will be updated:') . '</p>',
      ];
      $items = [];
      foreach ($confirm_data['fields'] as $field) {
        $items[] = $this->t(
          '@entity_type: <b>@field</b> (<i>@count references</i>)',
          [
            '@entity_type' => $field['entity_type'],
            '@field' => $field['field'],
            '@count' => $field['count'],
          ]
        );
      }
      $form['fields_list'] = [
        '#theme' => 'item_list',
        '#items' => $items,
      ];
      $form['confirm'] = [
        '#type' => 'submit',
        '#value' => $this->t('Confirm and execute merge'),
        '#button_type' => 'primary',
      ];
      $form['cancel'] = [
        '#type' => 'submit',
        '#value' => $this->t('Cancel'),
        '#limit_validation_errors' => [],
        '#submit' => ['::cancelSubmit'],
      ];
      return $form;
    }

    return $form;
  }

  /**
   * AJAX callback for rebuilding the form wrapper.
   */
  public function ajaxRebuild(array &$form, FormStateInterface $form_state): array {
    return $form['wrapper'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $sources = $form_state->getValue('source');
    $target = $form_state->getValue('target');

    // Ensure sources is an array
    // (entity_autocomplete with #tags=TRUE returns array).
    if (!is_array($sources)) {
      $sources = array_filter([$sources]);
    }

    if (in_array($target, $sources)) {
      $form_state->setErrorByName('source', $this->t('Source and target must be different entities.'));
    }
    if (empty($sources)) {
      $form_state->setErrorByName('source', $this->t('At least one source entity must be selected.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Handle confirmation step.
    if ($form_state->get('confirm_data')) {
      $data = $form_state->get('confirm_data');
      $entity_type = $data['entity_type'];
      $source_ids = $data['source_ids'];
      $target_id = $data['target_id'];
      $keep_source = $data['keep_source'];

      /** @var \Drupal\Core\Entity\EntityInterface $target */
      $target = $this->entityTypeManager->getStorage($entity_type)->load($target_id);

      foreach ($source_ids as $source_id) {
        if ($source_id == $target_id) {
          continue;
        }
        $source = $this->entityTypeManager->getStorage($entity_type)->load($source_id);
        if (!$source) {
          $this->messenger()->addError($this->t('Invalid source entity selected (ID: @id).', ['@id' => $source_id]));
          continue;
        }
        try {
          $this->mergeManager->execute($source, $target, [
            'keep_source' => $keep_source,
          ]);
          $this->messenger()->addStatus($this->t('Merge for source @source executed successfully.', ['@source' => $source_id]));
        }
        catch (\Throwable $e) {
          $this->messenger()->addError($this->t('Merge failed: @error', [
            '@error' => $e->getMessage(),
          ]));
        }
      }
      // Redirect to the same route after finishing.
      $current_path = $this->pathCurrent->getPath();
      $form_state->setRedirectUrl(Url::fromUserInput($current_path));
      return;
    }

    $entity_type = $form_state->getValue('entity_type');
    $source_ids = $form_state->getValue('source');
    $target_id = $form_state->getValue('target');
    $keep_source = (bool) $form_state->getValue('keep_source');

    foreach ($source_ids as $key => $value) {
      $source_ids[$key] = $value['target_id'];
    }
    if (!is_array($source_ids)) {
      $source_ids = array_filter([$source_ids]);
    }

    $target = $this->entityTypeManager->getStorage($entity_type)->load($target_id);
    if (!$target instanceof EntityInterface) {
      $this->messenger()->addError($this->t('Invalid target entity selected.'));
      return;
    }

    // Analyze fields and references for confirmation.
    $fields_info = [];
    foreach ($source_ids as $source_id) {
      if ($source_id == $target_id) {
        continue;
      }
      $source = $this->entityTypeManager->getStorage($entity_type)->load($source_id);
      if (!$source) {
        continue;
      }
      $result = $this->mergeManager->analyze($source, $target);
      if (!empty($result['fields'])) {
        foreach ($result['fields'] as $field) {
          $fields_info[] = $field;
        }
      }
    }

    if (empty($fields_info)) {
      $this->messenger()->addWarning($this->t('No references to update for the selected entities.'));
      return;
    }

    // Store confirmation data in form state.
    $form_state->set('confirm_data', [
      'entity_type' => $entity_type,
      'source_ids' => $source_ids,
      'target_id' => $target_id,
      'keep_source' => $keep_source,
      'fields' => $fields_info,
    ]);
    $form_state->setRebuild();
  }

  /**
   * Cancel confirmation and rebuild form.
   */
  public function cancelSubmit(array &$form, FormStateInterface $form_state): void {
    $form_state->set('confirm_data', NULL);
    $form_state->setRebuild();
  }

}
