<?php

namespace Drupal\entity_reference_manager\Handler;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Handles merging for node entities.
 */
class NodeMergeHandler implements ContentMergeHandlerInterface, ContainerInjectionInterface {
  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * Constructor.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    EntityFieldManagerInterface $entityFieldManager,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->entityFieldManager = $entityFieldManager;
  }

  /**
   * {@inheritdoc}
   */
  public function supports(EntityInterface $entity): bool {
    return $entity instanceof NodeInterface;
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(EntityInterface $source, EntityInterface $target): array {
    $sources = is_array($source) ? $source : [$source];
    $fieldMap = $this->entityFieldManager->getFieldMapByFieldType('entity_reference');

    $count = 0;
    $fields_info = [];

    foreach ($sources as $src) {
      foreach ($fieldMap as $entityType => $fields) {
        foreach ($fields as $fieldName => $info) {
          foreach ($info['bundles'] as $bundle) {
            $definitions = $this->entityFieldManager->getFieldDefinitions($entityType, $bundle);
            if (!isset($definitions[$fieldName])) {
              continue;
            }
            $definition = $definitions[$fieldName];
            $settings = $definition->getSettings();

            // Field must reference node.
            if (($settings['target_type'] ?? '') !== 'node') {
              continue;
            }

            // Check if target node bundle is accepted.
            if (!empty($settings['handler_settings']['target_bundles'])
              && !in_array($target->bundle(), $settings['handler_settings']['target_bundles'], TRUE)
            ) {
              continue;
            }

            $entities = $this->entityTypeManager->getStorage($entityType)->loadByProperties([
              $fieldName => $src->id(),
            ]);

            $field_count = 0;
            foreach ($entities as $entity) {
              foreach ($entity->get($fieldName)->getValue() as $value) {
                if ((int) $value['target_id'] === (int) $src->id()) {
                  $count++;
                  $field_count++;
                }
              }
            }
            if ($field_count > 0) {
              $fields_info[] = [
                'entity_type' => $entityType,
                'field' => $fieldName,
                'count' => $field_count,
              ];
            }
          }
        }
      }
    }

    return [
      'entity_type' => 'node',
      'source' => implode(',', array_map(fn($s) => $s->id(), $sources)),
      'target' => $target->id(),
      'fields' => $fields_info,
      'message' => $this->t('All references to the source node(s) will be updated. Total references to update: @count.', ['@count' => $count]),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function execute(EntityInterface $source, EntityInterface $target, array $options = []): void {
    $sources = is_array($source) ? $source : [$source];
    $fieldMap = $this->entityFieldManager->getFieldMapByFieldType('entity_reference');

    $operations = [];
    foreach ($sources as $src) {
      foreach ($fieldMap as $entityType => $fields) {
        foreach ($fields as $fieldName => $info) {
          foreach ($info['bundles'] as $bundle) {
            $definitions = $this->entityFieldManager->getFieldDefinitions($entityType, $bundle);
            if (!isset($definitions[$fieldName])) {
              continue;
            }
            $definition = $definitions[$fieldName];
            $settings = $definition->getSettings();
            if (($settings['target_type'] ?? '') !== 'node') {
              continue;
            }
            if (!empty($settings['handler_settings']['target_bundles'])
              && !in_array($target->bundle(), $settings['handler_settings']['target_bundles'], TRUE)
            ) {
              continue;
            }
            // Add batch operation for this field.
            $operations[] = [
              [static::class, 'batchUpdateFieldReferences'],
              [
                $entityType,
                $fieldName,
                $src->id(),
                $target->id(),
              ],
            ];
          }
        }
      }
      // Add operation to delete the source if needed.
      if (!($options['keep_source'] ?? FALSE)) {
        $operations[] = [
          [static::class, 'batchDeleteSource'],
          [$src->getEntityTypeId(), $src->id()],
        ];
      }
    }

    if (!empty($operations)) {
      $batch = [
        'title' => $this->t('Merging node references...'),
        'operations' => $operations,
        'finished' => [static::class, 'batchFinished'],
      ];
      batch_set($batch);
    }
  }

  /**
   * Batch operation to update field references.
   */
  public static function batchUpdateFieldReferences($entityType, $fieldName, $source_id, $target_id, array &$context) {
    $entityTypeManager = \Drupal::entityTypeManager();
    $entities = $entityTypeManager->getStorage($entityType)->loadByProperties([
      $fieldName => $source_id,
    ]);
    foreach ($entities as $entity) {
      if (!$entity->hasField($fieldName)) {
        continue;
      }
      $values = $entity->get($fieldName)->getValue();
      $changed = FALSE;
      foreach ($values as &$value) {
        if ((int) $value['target_id'] === (int) $source_id) {
          $value['target_id'] = $target_id;
          $changed = TRUE;
        }
      }
      if ($changed) {
        // Remove duplicates.
        $values = array_values(array_reduce($values, function ($carry, $item) {
          $carry[$item['target_id']] = $item;
          return $carry;
        }, []));
        $entity->set($fieldName, $values);
        $entity->save();
      }
    }
  }

  /**
   * Batch operation to delete the source entity.
   */
  public static function batchDeleteSource($entity_type_id, $source_id, array &$context) {
    $entityTypeManager = \Drupal::entityTypeManager();
    $entity = $entityTypeManager->getStorage($entity_type_id)->load($source_id);
    if ($entity) {
      $entity->delete();
    }
  }

  /**
   * Batch finished callback.
   */
  public static function batchFinished($success, $results, $operations) {
    $messenger = \Drupal::messenger();
    if ($success) {
      $messenger->addStatus(\Drupal::translation()->translate('Merge completed successfully.'));
    }
    else {
      $messenger->addError(\Drupal::translation()->translate('Merge batch did not complete.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager')
    );
  }

}
