<?php

namespace Drupal\entity_reference_manager\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Lock\LockBackendInterface;
use Drupal\entity_reference_manager\Handler\ContentMergeHandlerInterface;
use Psr\Log\LoggerInterface;

/**
 * Manages content merging operations.
 */
class ContentMergeManager {

  /**
   * Registered content merge handlers.
   *
   * @var \Drupal\entity_reference_manager\Handler\ContentMergeHandlerInterface[]
   */
  protected array $handlers = [];

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected Connection $database,
    protected LoggerInterface $logger,
    protected LockBackendInterface $lock,
    iterable $handlers = [],
  ) {
    foreach ($handlers as $handler) {
      $this->handlers[] = $handler;
    }
  }

  /**
   * Gets the appropriate handler for the given entity.
   */
  protected function getHandler(EntityInterface $entity): ?ContentMergeHandlerInterface {
    foreach ($this->handlers as $handler) {
      if ($handler->supports($entity)) {
        return $handler;
      }
    }
    return NULL;
  }

  /**
   * Analyzes the merge impact between source and target entities.
   */
  public function analyze(EntityInterface $source, EntityInterface $target): array {
    $handler = $this->getHandler($source);
    if (!$handler) {
      throw new \InvalidArgumentException('Unsupported entity type.');
    }
    return $handler->analyze($source, $target);
  }

  /**
   * Executes the merge between source and target entities.
   */
  public function execute(EntityInterface $source, EntityInterface $target, array $options = []): void {
    if (!$this->lock->acquire('entity_reference_manager')) {
      throw new \RuntimeException('Merge already running.');
    }

    $handler = $this->getHandler($source);
    if (!$handler) {
      throw new \InvalidArgumentException('Unsupported entity type.');
    }

    $this->database->startTransaction();
    try {
      $handler->execute($source, $target, $options);
      $this->lock->release('entity_reference_manager');
    }
    catch (\Throwable $e) {
      $this->lock->release('entity_reference_manager');
      throw $e;
    }
  }

}
