<?php

namespace Drupal\entity_reference_manager\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\entity_reference_manager\Service\ContentMergeManager;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for Entity Reference Manager.
 */
class EntityReferenceManagerCommands extends DrushCommands {

  /**
   * Entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Merge manager service.
   *
   * @var \Drupal\entity_reference_manager\Service\ContentMergeManager
   */
  protected $mergeManager;

  /**
   * EntityReferenceManagerCommands constructor.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, ContentMergeManager $mergeManager) {
    $this->entityTypeManager = $entityTypeManager;
    $this->mergeManager = $mergeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create($container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_reference_manager.manager')
    );
  }

  /**
   * Merge entity references via Drush.
   *
   * @command erm-merge
   *
   * @option entity_type The entity type (node, taxonomy_term, media)
   * @option source_ids  Comma-separated source entity IDs (e.g. 1,2,3)
   * @option target_id   Target entity ID
   * @option keep_source Set this flag to keep source entity after merge
   * @option update_revisions Set this flag to update references in revisions
   *
   * @usage drush erm-merge --entity_type=node --source_ids=1,2 --target_id=3 --keep_source --update_revisions
   */
  public function merge(
    $options = [
      'entity_type' => NULL,
      'source_ids' => NULL,
      'target_id' => NULL,
      'keep_source' => FALSE,
      'update_revisions' => FALSE,
    ],
  ) {
    // Use entity type definitions for a robust list.
    $definitions = $this->entityTypeManager->getDefinitions();
    $entity_types = [];
    foreach (['node', 'taxonomy_term', 'media'] as $type) {
      if (isset($definitions[$type])) {
        $entity_types[$type] = $definitions[$type]->getLabel();
      }
    }

    // Interactive selection with arrows, mapping index to machine name.
    $entity_type = $options['entity_type'] ?? NULL;
    if (!$entity_type) {
      $labels = array_values($entity_types);
      $keys = array_keys($entity_types);
      $selected_index = $this->io()->choice('Select entity type', $labels, 0);
      $entity_type = $keys[$selected_index];
    }
    elseif (!isset($entity_types[$entity_type])) {
      $this->io()->error('Invalid entity type.');
      return;
    }

    // Pergunta os IDs das entidades source.
    $source_ids = $options['source_ids'] ?? NULL;
    if (!$source_ids) {
      $source_ids = $this->io()->ask('Enter source entity IDs (comma separated)');
    }
    $source_ids = array_filter(array_map('trim', explode(',', $source_ids)));

    // Pergunta o ID do target.
    $target_id = $options['target_id'] ?? NULL;
    if (!$target_id) {
      $target_id = $this->io()->ask('Enter target entity ID');
    }

    // Opções extras.
    $keep_source = (bool) $options['keep_source'];
    $update_revisions = (bool) $options['update_revisions'];

    // Carrega target.
    $target = $this->entityTypeManager->getStorage($entity_type)->load($target_id);
    if (!$target) {
      $this->io()->error('Target entity not found.');
      return;
    }

    // Mostra análise antes de executar.
    foreach ($source_ids as $source_id) {
      if ($source_id == $target_id) {
        $this->io()->warning("Skipping source ID $source_id (same as target).");
        continue;
      }
      $source = $this->entityTypeManager->getStorage($entity_type)->load($source_id);
      if (!$source) {
        $this->io()->warning("Source entity $source_id not found.");
        continue;
      }

      // Chama analyze antes do merge.
      try {
        $analyze = $this->mergeManager->analyze($source, $target, [
          'update_revisions' => $update_revisions,
        ]);
        $this->io()->section("Analysis for source $source_id:");
        if (!empty($analyze['fields'])) {
          foreach ($analyze['fields'] as $field) {
            $msg = "- {$field['entity_type']}: {$field['field']} ({$field['count']} refs";
            if (!empty($field['count_revisions'])) {
              $msg .= ", {$field['count_revisions']} in revisions";
            }
            $msg .= ")";
            $this->io()->text($msg);
          }
        }
        else {
          $this->io()->note('No references to update for this source.');
        }
        if (!empty($analyze['message'])) {
          $this->io()->note($analyze['message']);
        }
      }
      catch (\Throwable $e) {
        $this->io()->warning("Analysis failed for source $source_id: " . $e->getMessage());
        continue;
      }
    }

    // Confirmação do usuário antes de executar.
    if (!$this->io()->confirm('Proceed with the merge operation?', TRUE)) {
      $this->io()->warning('Operation cancelled.');
      return;
    }

    // Executa para cada source.
    foreach ($source_ids as $source_id) {
      if ($source_id == $target_id) {
        continue;
      }
      $source = $this->entityTypeManager->getStorage($entity_type)->load($source_id);
      if (!$source) {
        continue;
      }
      try {
        $this->mergeManager->execute($source, $target, [
          'keep_source' => $keep_source,
          'update_revisions' => $update_revisions,
        ]);
        $this->io()->success("Merge for source $source_id executed successfully.");
      }
      catch (\Throwable $e) {
        $this->io()->error("Merge failed for source $source_id: " . $e->getMessage());
      }
    }
  }

}
