<?php

namespace Drupal\entity_reference_manager\Handler;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\media\MediaInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Handles merging for media entities.
 */
class MediaMergeHandler implements ContentMergeHandlerInterface, ContainerInjectionInterface {
  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * Constructor.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    EntityFieldManagerInterface $entityFieldManager,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->entityFieldManager = $entityFieldManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function supports(EntityInterface $entity): bool {
    return $entity instanceof MediaInterface;
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(EntityInterface $source, EntityInterface $target, array $options = []): array {
    // Accept array of sources for multi-merge support.
    $sources = is_array($source) ? $source : [$source];
    $fieldMap = $this->entityFieldManager->getFieldMapByFieldType('entity_reference');

    $count = 0;
    $fields_info = [];
    $update_revisions = $options['update_revisions'] ?? FALSE;

    foreach ($sources as $src) {
      foreach ($fieldMap as $entityType => $fields) {
        foreach ($fields as $fieldName => $info) {
          foreach ($info['bundles'] as $bundle) {
            $definitions = $this->entityFieldManager->getFieldDefinitions($entityType, $bundle);
            if (!isset($definitions[$fieldName])) {
              continue;
            }
            $definition = $definitions[$fieldName];
            $settings = $definition->getSettings();

            // Field must reference media.
            if (($settings['target_type'] ?? '') !== 'media') {
              continue;
            }

            // Check if target media bundle is accepted.
            if (!empty($settings['handler_settings']['target_bundles'])
              && !in_array($target->bundle(), $settings['handler_settings']['target_bundles'], TRUE)
            ) {
              continue;
            }

            // Entidades atuais.
            $entities = $this->entityTypeManager->getStorage($entityType)->loadByProperties([
              $fieldName => $src->id(),
            ]);

            $field_count = 0;
            foreach ($entities as $entity) {
              foreach ($entity->get($fieldName)->getValue() as $value) {
                if ((int) $value['target_id'] === (int) $src->id()) {
                  $count++;
                  $field_count++;
                }
              }
            }

            // Revisões, se solicitado.
            $field_count_revisions = 0;
            if ($update_revisions && $this->entityTypeManager->getDefinition($entityType)->isRevisionable()) {
              $storage = $this->entityTypeManager->getStorage($entityType);
              if (method_exists($storage, 'getQuery')) {
                $revision_ids = $storage->getQuery()
                  ->accessCheck(FALSE)
                  ->allRevisions()
                  ->condition($fieldName, $src->id())
                  ->execute();
                if (!empty($revision_ids)) {
                  $field_count_revisions = count($revision_ids);
                  $count += $field_count_revisions;
                }
              }
            }

            if ($field_count > 0 || $field_count_revisions > 0) {
              $fields_info[] = [
                'entity_type' => $entityType,
                'field' => $fieldName,
                'count' => $field_count,
                'count_revisions' => $field_count_revisions,
              ];
            }
          }
        }
      }
    }

    return [
      'entity_type' => 'media',
      'source' => implode(',', array_map(fn($s) => $s->id(), $sources)),
      'target' => $target->id(),
      'fields' => $fields_info,
      'message' => $this->t('All references to the source media will be replaced. Total references to update: @count.', ['@count' => $count]),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function execute(EntityInterface $source, EntityInterface $target, array $options = []): void {
    $sources = is_array($source) ? $source : [$source];
    $fieldMap = $this->entityFieldManager->getFieldMapByFieldType('entity_reference');
    $update_revisions = $options['update_revisions'] ?? FALSE;

    $operations = [];
    foreach ($sources as $src) {
      foreach ($fieldMap as $entityType => $fields) {
        foreach ($fields as $fieldName => $info) {
          foreach ($info['bundles'] as $bundle) {
            $definitions = $this->entityFieldManager->getFieldDefinitions($entityType, $bundle);
            if (!isset($definitions[$fieldName])) {
              continue;
            }
            $definition = $definitions[$fieldName];
            $settings = $definition->getSettings();
            if (($settings['target_type'] ?? '') !== 'media') {
              continue;
            }
            if (!empty($settings['handler_settings']['target_bundles'])
              && !in_array($target->bundle(), $settings['handler_settings']['target_bundles'], TRUE)
            ) {
              continue;
            }
            // Atualiza entidades atuais.
            $operations[] = [
              [static::class, 'batchUpdateFieldReferences'],
              [
                $entityType,
                $fieldName,
                $src->id(),
                $target->id(),
              ],
            ];
            // Atualiza revisões, se solicitado.
            if ($update_revisions && $this->entityTypeManager->getDefinition($entityType)->isRevisionable()) {
              $operations[] = [
                [static::class, 'batchUpdateFieldReferencesInRevisions'],
                [
                  $entityType,
                  $fieldName,
                  $src->id(),
                  $target->id(),
                ],
              ];
            }
          }
        }
      }
      if (!($options['keep_source'] ?? FALSE)) {
        $operations[] = [
          [static::class, 'batchDeleteSource'],
          [$src->getEntityTypeId(), $src->id()],
        ];
      }
    }

    if (!empty($operations)) {
      $batch = [
        'title' => $this->t('Merging media references...'),
        'operations' => $operations,
        'finished' => [static::class, 'batchFinished'],
      ];
      batch_set($batch);
    }
  }

  /**
   * Batch operation to update field references.
   */
  public static function batchUpdateFieldReferences($entityType, $fieldName, $source_id, $target_id, array &$context) {
    $entityTypeManager = \Drupal::entityTypeManager();
    $entities = $entityTypeManager->getStorage($entityType)->loadByProperties([
      $fieldName => $source_id,
    ]);
    foreach ($entities as $entity) {
      if (!$entity->hasField($fieldName)) {
        continue;
      }
      $values = $entity->get($fieldName)->getValue();
      $changed = FALSE;
      foreach ($values as &$value) {
        if ((int) $value['target_id'] === (int) $source_id) {
          $value['target_id'] = $target_id;
          $changed = TRUE;
        }
      }
      if ($changed) {
        // Remove duplicate target_id.
        $values = array_values(array_reduce($values, function ($carry, $item) {
          $carry[$item['target_id']] = $item;
          return $carry;
        }, []));
        $entity->set($fieldName, $values);
        $entity->save();
      }
    }
  }

  /**
   * Batch operation to update field references in revisions.
   */
  public static function batchUpdateFieldReferencesInRevisions($entityType, $fieldName, $source_id, $target_id, array &$context) {
    $entityTypeManager = \Drupal::entityTypeManager();
    $definition = $entityTypeManager->getDefinition($entityType);
    if (!$definition->isRevisionable()) {
      return;
    }
    $storage = $entityTypeManager->getStorage($entityType);
    if (!method_exists($storage, 'getQuery')) {
      return;
    }
    $revision_ids = $storage->getQuery()->accessCheck(FALSE)
      ->allRevisions()
      ->condition($fieldName, $source_id)
      ->execute();
    if (empty($revision_ids)) {
      return;
    }
    foreach ($revision_ids as $revision_id => $entity_id) {
      $entity = $storage->loadRevision($revision_id);
      if (!$entity || !$entity->hasField($fieldName)) {
        continue;
      }
      $values = $entity->get($fieldName)->getValue();
      $changed = FALSE;
      foreach ($values as &$value) {
        if ((int) $value['target_id'] === (int) $source_id) {
          $value['target_id'] = $target_id;
          $changed = TRUE;
        }
      }
      if ($changed) {
        // Remove duplicate target_id.
        $values = array_values(array_reduce($values, function ($carry, $item) {
          $carry[$item['target_id']] = $item;
          return $carry;
        }, []));
        $entity->set($fieldName, $values);

        if (method_exists($entity, 'setNewRevision')) {
          $entity->setNewRevision(FALSE);
        }
        $entity->save();
      }
    }
  }

  /**
   * Batch operation to delete the source entity.
   */
  public static function batchDeleteSource($entity_type_id, $source_id, array &$context) {
    $entityTypeManager = \Drupal::entityTypeManager();
    $entity = $entityTypeManager->getStorage($entity_type_id)->load($source_id);
    if ($entity) {
      $entity->delete();
    }
  }

  /**
   * Batch finished callback.
   */
  public static function batchFinished($success, $results, $operations) {
    $messenger = \Drupal::messenger();
    if ($success) {
      $messenger->addStatus(\Drupal::translation()->translate('Merge completed successfully.'));
    }
    else {
      $messenger->addError(\Drupal::translation()->translate('Merge batch did not complete.'));
    }
  }

}
