<?php

namespace Drupal\entity_reservation_system\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * Provides a form for deleting a Reservation unit revision.
 *
 * @ingroup entity_reservation_system
 */
class ReservationUnitRevisionDeleteForm extends ConfirmFormBase {


  /**
   * The Reservation unit revision.
   *
   * @var \Drupal\entity_reservation_system\Entity\ReservationUnitInterface
   */
  protected $revision;

  /**
   * The Reservation unit storage.
   *
   * @var \Drupal\Core\Entity\RevisionableStorageInterface
   */
  protected $reservationUnitStorage;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new ReservationUnitRevisionDeleteForm.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $entity_storage
   *   The entity storage.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger interface.
   */
  public function __construct(
    EntityStorageInterface $entity_storage,
    Connection $connection,
    MessengerInterface $messenger
  ) {
    $this->reservationUnitStorage = $entity_storage;
    $this->connection = $connection;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $entity_manager = $container->get('entity_type.manager');
    return new static(
      $entity_manager->getStorage('reservation_unit'),
      $container->get('database'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'reservation_unit_revision_delete_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return t('Are you sure you want to delete the revision from %revision-date?', ['%revision-date' => Drupal::service('date.formatter')->format($this->revision->getRevisionCreationTime())]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('entity.reservation_unit.version_history', ['reservation_unit' => $this->revision->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $reservation_unit_revision = NULL) {
    $this->revision = $this->reservationUnitStorage->loadRevision($reservation_unit_revision);
    $form = parent::buildForm($form, $form_state);

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->reservationUnitStorage->deleteRevision($this->revision->getRevisionId());

    $this->logger('content')->notice('Reservation unit: deleted %title revision %revision.', ['%title' => $this->revision->label(), '%revision' => $this->revision->getRevisionId()]);
    $this->messenger->addStatus(t('Revision from %revision-date of Reservation unit %title has been deleted.', ['%revision-date' => Drupal::service('date.formatter')->format($this->revision->getRevisionCreationTime()), '%title' => $this->revision->label()]));
    $form_state->setRedirect(
      'entity.reservation_unit.canonical',
       ['reservation_unit' => $this->revision->id()]
    );
    if ($this->connection->query('SELECT COUNT(DISTINCT vid) FROM {reservation_unit_field_revision} WHERE id = :id', [':id' => $this->revision->id()])->fetchField() > 1) {
      $form_state->setRedirect(
        'entity.reservation_unit.version_history',
         ['reservation_unit' => $this->revision->id()]
      );
    }
  }

}
