<?php

declare(strict_types=1);

namespace Drupal\entity_revision_diff;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides dynamic permissions for revisionable entity bundles.
 */
class EntityDiffPermissions implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * Constructs an EntityDiffPermissions object.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $bundleInfo,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
    );
  }

  /**
   * Entity types that need dynamic global permissions.
   *
   * These are entity types from optional modules (like group)
   * that shouldn't have static permissions in .permissions.yml.
   *
   * @var array
   */
  protected const DYNAMIC_PERMISSION_ENTITY_TYPES = ['group'];

  /**
   * Returns an array of bundle-specific revision permissions.
   *
   * @return array
   *   The permissions array.
   */
  public function bundlePermissions(): array {
    $perms = [];
    $supported_types = entity_revision_diff_supported_entity_types();
    foreach (array_keys($supported_types) as $entity_type_id) {
      try {
        $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      }
      catch (\Exception $e) {
        continue;
      }
      if (!$entity_type->isRevisionable()) {
        continue;
      }
      // Add dynamic global permissions for optional entity types.
      if (in_array($entity_type_id, self::DYNAMIC_PERMISSION_ENTITY_TYPES, TRUE)) {
        $perms += $this->buildGlobalPermissions($entity_type_id, (string) $entity_type->getLabel());
      }
      $bundles = $this->bundleInfo->getBundleInfo($entity_type_id);
      $entity_type_label = $entity_type->getLabel();
      foreach ($bundles as $bundle_id => $bundle_info) {
        $bundle_label = $bundle_info['label'] ?? $bundle_id;
        $perms += $this->buildBundlePermissions($entity_type_id, $bundle_id, $entity_type_label, $bundle_label);
      }
    }

    return $perms;
  }

  /**
   * Builds global permissions for an entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $entity_type_label
   *   The entity type label.
   *
   * @return array
   *   An associative array of permission names and descriptions.
   */
  protected function buildGlobalPermissions(string $entity_type_id, string $entity_type_label): array {
    return [
      "view all {$entity_type_id} revisions" => [
        'title' => $this->t('View all @type revisions', ['@type' => $entity_type_label]),
        'description' => $this->t('To view a revision, you also need permission to view the @type.', ['@type' => $entity_type_label]),
      ],
      "revert all {$entity_type_id} revisions" => [
        'title' => $this->t('Revert all @type revisions', ['@type' => $entity_type_label]),
        'description' => $this->t('To revert a revision, you also need permission to edit the @type.', ['@type' => $entity_type_label]),
      ],
      "delete all {$entity_type_id} revisions" => [
        'title' => $this->t('Delete all @type revisions', ['@type' => $entity_type_label]),
        'description' => $this->t('To delete a revision, you also need permission to delete the @type.', ['@type' => $entity_type_label]),
      ],
    ];
  }

  /**
   * Builds permissions for a specific entity bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param string $entity_type_label
   *   The entity type label.
   * @param string $bundle_label
   *   The bundle label.
   *
   * @return array
   *   An associative array of permission names and descriptions.
   */
  protected function buildBundlePermissions(string $entity_type_id, string $bundle_id, $entity_type_label, $bundle_label): array {
    $params = [
      '%bundle' => $bundle_label,
      '%entity_type' => $entity_type_label,
    ];
    return [
      "view $bundle_id revisions" => [
        'title' => $this->t('%bundle %entity_type: View revisions', $params),
        'description' => $this->t('To view a revision, you also need permission to view the entity.'),
      ],
      "revert $bundle_id revisions" => [
        'title' => $this->t('%bundle %entity_type: Revert revisions', $params),
        'description' => $this->t('To revert a revision, you also need permission to edit the entity.'),
      ],
      "delete $bundle_id revisions" => [
        'title' => $this->t('%bundle %entity_type: Delete revisions', $params),
        'description' => $this->t('To delete a revision, you also need permission to delete the entity.'),
      ],
    ];
  }

}
