<?php

declare(strict_types=1);

namespace Drupal\entity_revision_diff\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\RevisionLogInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for reverting an entity revision for a single translation.
 */
class EntityRevisionRevertTranslationForm extends ConfirmFormBase {

  /**
   * The entity revision.
   */
  protected ContentEntityInterface $revision;

  /**
   * The entity type ID.
   */
  protected string $entityTypeId;

  /**
   * The language to be reverted.
   */
  protected string $langcode;

  /**
   * Constructs a new EntityRevisionRevertTranslationForm.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected DateFormatterInterface $dateFormatter,
    protected LanguageManagerInterface $languageManager,
    protected TimeInterface $time,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('date.formatter'),
      $container->get('language_manager'),
      $container->get('datetime.time'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'gsm_entity_revision_revert_translation_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    $revision_date = $this->revision instanceof RevisionLogInterface
      ? $this->dateFormatter->format($this->revision->getRevisionCreationTime())
      : $this->t('unknown date');
    return $this->t('Are you sure you want to revert @language translation to the revision from %revision-date?', [
      '@language' => $this->languageManager->getLanguageName($this->langcode),
      '%revision-date' => $revision_date,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url("entity.{$this->entityTypeId}.version_history", [$this->entityTypeId => $this->revision->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Revert');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $langcode = NULL): array {
    // Get entity type and revision from route.
    $route_match = $this->getRouteMatch();
    $route_name = $route_match->getRouteName();
    if ($route_name && preg_match('/entity_revision_diff\.(\w+)_revision_revert_translation/', $route_name, $matches)) {
      $this->entityTypeId = $matches[1];
    }
    else {
      $this->entityTypeId = 'unknown';
    }

    // Get the revision from the route parameter.
    // Parameter name format: {entity_type}_revision.
    $revision_param_name = $this->entityTypeId . '_revision';
    $entity_revision = $route_match->getParameter($revision_param_name);
    if (!$entity_revision instanceof ContentEntityInterface) {
      throw new \InvalidArgumentException('Entity revision not found in route parameters');
    }

    $this->langcode = $langcode ?? '';
    $this->revision = $entity_revision;
    $form = parent::buildForm($form, $form_state);
    // Unless untranslatable fields are configured to affect only the default
    // translation, we need to ask the user whether they should be included.
    $default_translation_affected = $this->revision->isDefaultTranslationAffectedOnly();
    $form['revert_untranslated_fields'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Revert content shared among translations'),
      '#default_value' => $default_translation_affected && $this->revision->getTranslation($this->langcode)->isDefaultTranslation(),
      '#access' => !$default_translation_affected,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $revert_untranslated_fields = (bool) $form_state->getValue('revert_untranslated_fields');
    $translation = $this->revision->getTranslation($this->langcode);
    $storage = $this->entityTypeManager->getStorage($this->entityTypeId);
    // The revision timestamp will be updated when the revision is saved.
    $new_revision = $storage->createRevision($translation, TRUE, $revert_untranslated_fields);

    if ($new_revision instanceof RevisionLogInterface) {
      $revision_date = $this->revision instanceof RevisionLogInterface
        ? $this->dateFormatter->format($this->revision->getRevisionCreationTime())
        : $this->t('unknown date');
      $new_revision->setRevisionLogMessage($this->t('Copy of the revision from %date.', [
        '%date' => $revision_date,
      ]));
      $new_revision->setRevisionUserId($this->currentUser()->id());
      $new_revision->setRevisionCreationTime($this->time->getRequestTime());
    }
    if (method_exists($new_revision, 'setChangedTime')) {
      $new_revision->setChangedTime($this->time->getRequestTime());
    }
    $new_revision->save();
    $entity_type_label = $this->entityTypeManager->getDefinition($this->entityTypeId)?->getLabel() ?? $this->entityTypeId;
    $this->logger('content')->notice('@type: reverted @language translation of %title to revision %revision.', [
      '@type' => $entity_type_label,
      '@language' => $this->languageManager->getLanguageName($this->langcode),
      '%title' => $this->revision->label(),
      '%revision' => $this->revision->getRevisionId(),
    ]);
    $this->messenger()->addStatus($this->t('@type %title @language translation has been reverted to the revision from %revision-date.', [
      '@type' => $entity_type_label,
      '%title' => $this->revision->label(),
      '@language' => $this->languageManager->getLanguageName($this->langcode),
      '%revision-date' => $this->revision instanceof RevisionLogInterface
        ? $this->dateFormatter->format($this->revision->getRevisionCreationTime())
        : $this->t('unknown date'),
    ]));
    $form_state->setRedirect("entity.{$this->entityTypeId}.version_history", [$this->entityTypeId => $this->revision->id()]);
  }

}
