<?php

declare(strict_types=1);

namespace Drupal\Tests\entity_revision_diff\Kernel;

/**
 * Tests views data provided by entity_revision_diff module.
 *
 * These tests will fail if:
 * - Views module changes views data API
 * - Core changes entity revision table naming
 * - Module changes views field plugin ID
 *
 * @group entity_revision_diff
 */
class ViewsDataTest extends EntityRevisionDiffKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'filter',
    'node',
    'diff',
    'block',
    'block_content',
    'media',
    'image',
    'file',
    'taxonomy',
    'entity',
    'flexible_permissions',
    'group',
    'options',
    'variationcache',
    'views',
    'entity_revision_diff',
  ];

  /**
   * The views data service.
   *
   * @var \Drupal\views\ViewsData
   */
  protected $viewsData;

  /**
   * {@inheritdoc}
   * @throws \Exception
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('block_content');
    $this->installEntitySchema('media');
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('taxonomy_vocabulary');
    $this->installEntitySchema('group');
    $this->installEntitySchema('group_content');
    $this->installEntitySchema('group_config_wrapper');
    $this->installEntitySchema('file');
    $this->installConfig(['block_content', 'media', 'taxonomy', 'group', 'views']);
    $this->viewsData = $this->container->get('views.views_data');
  }

  /**
   * Tests that gsm_entity_current_vid field is added to group revision table.
   *
   * @throws \Exception
   */
  public function testGroupRevisionViewsField(): void {
    $data = $this->viewsData->get('groups_field_revision');
    $this->assertArrayHasKey(
      'gsm_entity_current_vid',
      $data,
      'gsm_entity_current_vid field must exist in groups_field_revision table'
    );
    $field_data = $data['gsm_entity_current_vid'];
    $this->assertArrayHasKey('title', $field_data);
    $this->assertArrayHasKey('help', $field_data);
    $this->assertArrayHasKey('field', $field_data);
    $this->assertEquals('gsm_entity_current_vid', $field_data['field']['id']);
  }

  /**
   * Tests that gsm_entity_current_vid field is added to block_content revision
   * table.
   *
   * @throws \Exception
   */
  public function testBlockContentRevisionViewsField(): void {
    $data = $this->viewsData->get('block_content_field_revision');
    $this->assertArrayHasKey(
      'gsm_entity_current_vid',
      $data,
      'gsm_entity_current_vid field must exist in block_content_field_revision table'
    );
    $field_data = $data['gsm_entity_current_vid'];
    $this->assertEquals('gsm_entity_current_vid', $field_data['field']['id']);
  }

  /**
   * Tests that gsm_entity_current_vid field is added to media revision table.
   *
   * @throws \Exception
   */
  public function testMediaRevisionViewsField(): void {
    $data = $this->viewsData->get('media_field_revision');
    $this->assertArrayHasKey(
      'gsm_entity_current_vid',
      $data,
      'gsm_entity_current_vid field must exist in media_field_revision table'
    );
    $field_data = $data['gsm_entity_current_vid'];
    $this->assertEquals('gsm_entity_current_vid', $field_data['field']['id']);
  }

  /**
   * Tests that gsm_entity_current_vid field is added to taxonomy_term revision
   * table.
   *
   * @throws \Exception
   */
  public function testTaxonomyTermRevisionViewsField(): void {
    $data = $this->viewsData->get('taxonomy_term_field_revision');
    $this->assertArrayHasKey(
      'gsm_entity_current_vid',
      $data,
      'gsm_entity_current_vid field must exist in taxonomy_term_field_revision table'
    );
    $field_data = $data['gsm_entity_current_vid'];
    $this->assertEquals('gsm_entity_current_vid', $field_data['field']['id']);
  }

  /**
   * Tests field data structure is correct.
   *
   * @dataProvider revisionTablesProvider
   * @throws \Exception
   */
  public function testFieldDataStructure(string $table_name): void {
    $data = $this->viewsData->get($table_name);
    if (!isset($data['gsm_entity_current_vid'])) {
      $this->markTestSkipped("Table $table_name does not have gsm_entity_current_vid field");
    }
    $field_data = $data['gsm_entity_current_vid'];
    // Required keys.
    $this->assertArrayHasKey('title', $field_data);
    $this->assertArrayHasKey('help', $field_data);
    $this->assertArrayHasKey('field', $field_data);
    // Field must have id.
    $this->assertArrayHasKey('id', $field_data['field']);
    // Title and help should be translatable.
    $this->assertNotEmpty($field_data['title']);
    $this->assertNotEmpty($field_data['help']);
  }

  /**
   * Data provider for revision tables.
   *
   * @return array
   *   Test data.
   */
  public static function revisionTablesProvider(): array {
    return [
      'groups_field_revision' => ['groups_field_revision'],
      'block_content_field_revision' => ['block_content_field_revision'],
      'media_field_revision' => ['media_field_revision'],
      'taxonomy_term_field_revision' => ['taxonomy_term_field_revision'],
    ];
  }

}
