<?php

declare(strict_types=1);

namespace Drupal\Tests\entity_revision_diff\Unit;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\entity_revision_diff\Plugin\views\field\EntityCurrentRevisionVID;
use Drupal\Tests\UnitTestCase;
use Drupal\views\ResultRow;

/**
 * Unit tests for EntityCurrentRevisionVID views field plugin.
 *
 * These tests will fail if:
 * - Views module changes ResultRow structure
 * - Core changes entity interface for getRevisionId
 * - Core changes entity storage load API
 *
 * @group entity_revision_diff
 * @coversDefaultClass \Drupal\entity_revision_diff\Plugin\views\field\EntityCurrentRevisionVID
 */
class EntityCurrentRevisionVIDTest extends UnitTestCase {

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The plugin instance under test.
   *
   * @var \Drupal\entity_revision_diff\Plugin\views\field\EntityCurrentRevisionVID
   */
  protected $plugin;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $configuration = [];
    $plugin_id = 'gsm_entity_current_vid';
    $plugin_definition = ['id' => 'gsm_entity_current_vid'];
    $this->plugin = new EntityCurrentRevisionVID(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $this->entityTypeManager
    );
  }

  /**
   * Tests render method returns empty string when no entity.
   *
   * @covers ::render
   */
  public function testRenderReturnsEmptyWhenNoEntity(): void {
    $result_row = new ResultRow();
    // No _entity property set.
    $result = $this->plugin->render($result_row);
    $this->assertEquals('', $result);
  }

  /**
   * Tests render method returns empty string when entity is null.
   *
   * @covers ::render
   */
  public function testRenderReturnsEmptyWhenEntityNull(): void {
    $result_row = new ResultRow();
    $result_row->_entity = NULL;
    $result = $this->plugin->render($result_row);
    $this->assertEquals('', $result);
  }

  /**
   * Tests render method returns revision ID for valid entity.
   *
   * @covers ::render
   */
  public function testRenderReturnsRevisionIdForValidEntity(): void {
    $entity_type_id = 'group';
    $entity_id = 123;
    $revision_id = 456;
    // Create mock entity for result row.
    $revision_entity = $this->createMock(ContentEntityInterface::class);
    $revision_entity->method('getEntityTypeId')->willReturn($entity_type_id);
    $revision_entity->method('id')->willReturn($entity_id);
    // Create mock current entity with revision ID.
    $current_entity = $this->createMock(ContentEntityInterface::class);
    $current_entity->method('getRevisionId')->willReturn($revision_id);
    // Create mock storage.
    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->method('load')
      ->with($entity_id)
      ->willReturn($current_entity);
    $this->entityTypeManager->method('getStorage')
      ->with($entity_type_id)
      ->willReturn($storage);
    $result_row = new ResultRow();
    $result_row->_entity = $revision_entity;
    $result = $this->plugin->render($result_row);
    $this->assertEquals('456', $result);
  }

  /**
   * Tests render method returns empty when current entity not found.
   *
   * @covers ::render
   */
  public function testRenderReturnsEmptyWhenCurrentEntityNotFound(): void {
    $entity_type_id = 'group';
    $entity_id = 123;
    // Create mock entity for result row.
    $revision_entity = $this->createMock(ContentEntityInterface::class);
    $revision_entity->method('getEntityTypeId')->willReturn($entity_type_id);
    $revision_entity->method('id')->willReturn($entity_id);
    // Create mock storage that returns null.
    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->method('load')
      ->with($entity_id)
      ->willReturn(NULL);
    $this->entityTypeManager->method('getStorage')
      ->with($entity_type_id)
      ->willReturn($storage);
    $result_row = new ResultRow();
    $result_row->_entity = $revision_entity;
    $result = $this->plugin->render($result_row);
    $this->assertEquals('', $result);
  }

  /**
   * Tests usesGroupBy returns FALSE.
   *
   * @covers ::usesGroupBy
   */
  public function testUsesGroupByReturnsFalse(): void {
    $this->assertFalse($this->plugin->usesGroupBy());
  }

  /**
   * Tests query method does nothing (override parent).
   *
   * @covers ::query
   */
  public function testQueryDoesNothing(): void {
    // Just verify it doesn't throw an error.
    $this->plugin->query();
    $this->assertTrue(TRUE);
  }

}
