<?php

declare(strict_types=1);

namespace Drupal\entity_revision_diff\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Entity\RevisionLogInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for displaying entity revisions.
 */
class EntityRevisionController extends ControllerBase {

  /**
   * Constructs an EntityRevisionController object.
   *
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   */
  public function __construct(
    protected DateFormatterInterface $dateFormatter,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('date.formatter'),
    );
  }

  /**
   * Displays a group revision.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $group_revision
   *   The group revision.
   *
   * @return array
   *   A render array.
   */
  public function showGroupRevision(RevisionableInterface $group_revision): array {
    return $this->buildRevisionView($group_revision, 'group');
  }

  /**
   * Displays a block_content revision.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $block_content_revision
   *   The block_content revision.
   *
   * @return array
   *   A render array.
   */
  public function showBlockContentRevision(RevisionableInterface $block_content_revision): array {
    return $this->buildRevisionView($block_content_revision, 'block_content');
  }

  /**
   * Displays a media revision.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $media_revision
   *   The media revision.
   *
   * @return array
   *   A render array.
   */
  public function showMediaRevision(RevisionableInterface $media_revision): array {
    return $this->buildRevisionView($media_revision, 'media');
  }

  /**
   * Displays a taxonomy_term revision.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $taxonomy_term_revision
   *   The taxonomy_term revision.
   *
   * @return array
   *   A render array.
   */
  public function showTaxonomyTermRevision(RevisionableInterface $taxonomy_term_revision): array {
    return $this->buildRevisionView($taxonomy_term_revision, 'taxonomy_term');
  }

  /**
   * Build the revision view.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $entity
   *   The entity revision.
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return array
   *   A render array.
   */
  protected function buildRevisionView(RevisionableInterface $entity, string $entity_type_id): array {
    $view_builder = $this->entityTypeManager()->getViewBuilder($entity_type_id);
    return $view_builder->view($entity, 'full');
  }

  /**
   * Page title callback for group revision.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $group_revision
   *   The group revision.
   *
   * @return string
   *   The page title.
   */
  public function groupRevisionTitle(RevisionableInterface $group_revision): string {
    return $this->revisionTitle($group_revision);
  }

  /**
   * Page title callback for block_content revision.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $block_content_revision
   *   The block_content revision.
   *
   * @return string
   *   The page title.
   */
  public function blockContentRevisionTitle(RevisionableInterface $block_content_revision): string {
    return $this->revisionTitle($block_content_revision);
  }

  /**
   * Page title callback for media revision.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $media_revision
   *   The media revision.
   *
   * @return string
   *   The page title.
   */
  public function mediaRevisionTitle(RevisionableInterface $media_revision): string {
    return $this->revisionTitle($media_revision);
  }

  /**
   * Page title callback for taxonomy_term revision.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $taxonomy_term_revision
   *   The taxonomy_term revision.
   *
   * @return string
   *   The page title.
   */
  public function taxonomyTermRevisionTitle(RevisionableInterface $taxonomy_term_revision): string {
    return $this->revisionTitle($taxonomy_term_revision);
  }

  /**
   * Build the revision title.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $entity
   *   The entity revision.
   *
   * @return string
   *   The page title.
   */
  protected function revisionTitle(RevisionableInterface $entity): string {
    if ($entity instanceof RevisionLogInterface) {
      return (string) $this->t('Revision of %title from %date', [
        '%title' => $entity->label(),
        '%date' => $this->dateFormatter->format($entity->getRevisionCreationTime()),
      ]);
    }
    return (string) $this->t('Revision');
  }

}
