<?php

declare(strict_types=1);

namespace Drupal\entity_revision_diff\Form;

use Drupal\Component\Utility\Xss;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\RevisionLogInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\diff\DiffEntityComparison;
use Drupal\diff\DiffLayoutManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a universal form for entity revision overview with diff comparison.
 */
class EntityRevisionOverviewForm extends FormBase {

  /**
   * The diff settings config.
   */
  protected ImmutableConfig $config;

  /**
   * Constructs an EntityRevisionOverviewForm object.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected AccountInterface $currentUser,
    protected DateFormatterInterface $dateFormatter,
    protected RendererInterface $renderer,
    protected LanguageManagerInterface $languageManager,
    protected DiffLayoutManager $diffLayoutManager,
    protected DiffEntityComparison $entityComparison,
    protected TimeInterface $time,
  ) {
    $this->config = $this->config('diff.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('current_user'),
      $container->get('date.formatter'),
      $container->get('renderer'),
      $container->get('language_manager'),
      $container->get('plugin.manager.diff.layout'),
      $container->get('diff.entity_comparison'),
      $container->get('datetime.time'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'entity_revision_diff_overview_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?ContentEntityInterface $entity = NULL): array {
    // Try to get entity from route if not provided directly.
    if ($entity === NULL) {
      $entity = $this->getEntityFromRoute();
    }
    if ($entity === NULL) {
      return $form;
    }
    $entity_type_id = $entity->getEntityTypeId();
    $entity_type = $entity->getEntityType();
    $langcode = $entity->language()->getId();
    $langname = $entity->language()->getName();
    $languages = $entity->getTranslationLanguages();
    $has_translations = (\count($languages) > 1);
    $storage = $this->entityTypeManager->getStorage($entity_type_id);
    $vids = $this->getRevisionIds($entity);
    $revision_count = \count($vids);
    $build['#title'] = $has_translations ? $this->t('@langname revisions for %title', [
      '@langname' => $langname,
      '%title' => $entity->label(),
    ]) : $this->t('Revisions for %title', [
      '%title' => $entity->label(),
    ]);
    $build['entity_id'] = [
      '#type' => 'hidden',
      '#value' => $entity->id(),
    ];
    $build['entity_type_id'] = [
      '#type' => 'hidden',
      '#value' => $entity_type_id,
    ];
    $table_header = [];
    $table_header['revision'] = $this->t('Revision information');
    // Allow comparisons only if there are 2 or more revisions.
    $table_caption = '';
    if ($revision_count > 1) {
      $table_caption = $this->t('Use the radio buttons in the table below to select two revisions to compare. Then click the "Compare selected revisions" button to generate the comparison.');
      $table_header += [
        'select_column_one' => $this->t('Source revision'),
        'select_column_two' => $this->t('Target revision'),
      ];
    }
    $table_header['operations'] = $this->t('Operations');
    // Check permissions based on entity type.
    $bundle = $entity->bundle();
    $revert_permission = $this->checkRevisionPermission($entity, 'revert', $bundle);
    $delete_permission = $this->checkRevisionPermission($entity, 'delete', $bundle);
    // Submit button for the form.
    $compare_revision_submit = [
      '#type' => 'submit',
      '#button_type' => 'primary',
      '#value' => $this->t('Compare selected revisions'),
      '#attributes' => [
        'class' => ['diff-button'],
      ],
    ];
    // For more than 5 revisions, add a submit button on top.
    if ($revision_count > 5) {
      $build['submit_top'] = $compare_revision_submit;
    }
    // Contains the table listing the revisions.
    $build['entity_revisions_table'] = [
      '#type' => 'table',
      '#caption' => $table_caption,
      '#header' => $table_header,
      '#attributes' => ['class' => ['diff-revisions']],
    ];
    $build['entity_revisions_table']['#attached']['library'][] = 'diff/diff.general';
    $build['entity_revisions_table']['#attached']['drupalSettings']['diffRevisionRadios'] = $this->config->get('general_settings.radio_behavior');
    $default_revision = $entity->getRevisionId();
    // Add rows to the table.
    foreach ($vids as $key => $vid) {
      $previous_revision = NULL;
      if (isset($vids[$key + 1])) {
        $previous_revision = $storage->loadRevision($vids[$key + 1]);
      }
      /** @var \Drupal\Core\Entity\ContentEntityInterface $revision */
      $revision = $storage->loadRevision($vid);
      if ($revision === NULL) {
        continue;
      }
      if ($revision->hasTranslation($langcode) && $revision->getTranslation($langcode)->isRevisionTranslationAffected()) {
        $username = [];
        if ($revision instanceof RevisionLogInterface) {
          $username = [
            '#theme' => 'username',
            '#account' => $revision->getRevisionUser(),
          ];
        }
        $revision_date = $this->dateFormatter->format(
          $revision instanceof RevisionLogInterface ? $revision->getRevisionCreationTime() : $this->time->getRequestTime(),
          'short'
        );
        // Use revision link for non-current revisions.
        if ($vid != $entity->getRevisionId() && $entity_type->hasLinkTemplate('revision')) {
          $link = Link::fromTextAndUrl($revision_date, $revision->toUrl('revision'));
        }
        else {
          $link = $entity->toLink($revision_date);
        }
        if ($vid == $default_revision) {
          $row = [
            'revision' => $this->buildRevision($link, $username, $revision, $previous_revision),
          ];
          // Allow comparisons only if there are 2 or more revisions.
          if ($revision_count > 1) {
            $row += [
              'select_column_one' => $this->buildSelectColumn('radios_left', $vid, FALSE),
              'select_column_two' => $this->buildSelectColumn('radios_right', $vid, $vid),
            ];
          }
          $row['operations'] = [
            '#prefix' => '<em>',
            '#markup' => $this->t('Current revision'),
            '#suffix' => '</em>',
            '#attributes' => [
              'class' => ['revision-current'],
            ],
          ];
          $row['#attributes'] = [
            'class' => ['revision-current'],
          ];
        }
        else {
          $links = $this->buildOperationLinks($entity, $revision, $vid, $has_translations, $langcode, $revert_permission, $delete_permission);
          $row = [
            'revision' => $this->buildRevision($link, $username, $revision, $previous_revision),
            'select_column_one' => $this->buildSelectColumn('radios_left', $vid, $vids[1] ?? FALSE),
            'select_column_two' => $this->buildSelectColumn('radios_right', $vid, FALSE),
            'operations' => [
              '#type' => 'operations',
              '#links' => $links,
            ],
          ];
        }
        // Add the row to the table.
        $build['entity_revisions_table'][] = $row;
      }
    }
    // Allow comparisons only if there are 2 or more revisions.
    if ($revision_count > 1) {
      $build['submit'] = $compare_revision_submit;
    }
    $build['pager'] = ['#type' => 'pager'];
    $form_state->set('workspace_safe', TRUE);
    return $build;
  }

  /**
   * Gets entity from current route.
   */
  protected function getEntityFromRoute(): ?ContentEntityInterface {
    $route_match = $this->getRouteMatch();
    $route = $route_match->getRouteObject();
    if ($route === NULL) {
      return NULL;
    }
    $entity_type_id = $route->getOption('entity_type_id');
    if ($entity_type_id) {
      $entity = $route_match->getParameter($entity_type_id);
      if ($entity instanceof ContentEntityInterface) {
        return $entity;
      }
    }
    // Try to find entity from route parameters.
    foreach ($route_match->getParameters()->all() as $param) {
      if ($param instanceof ContentEntityInterface && $param->getEntityType()->isRevisionable()) {
        return $param;
      }
    }
    return NULL;
  }

  /**
   * Checks revision permission for an entity.
   */
  protected function checkRevisionPermission(ContentEntityInterface $entity, string $operation, string $bundle): bool {
    $entity_type_id = $entity->getEntityTypeId();
    $account = $this->currentUser;
    // Check generic permissions.
    if ($account->hasPermission("$operation all $entity_type_id revisions") ||
        $account->hasPermission("administer $entity_type_id") ||
        $account->hasPermission('administer content')) {
      return TRUE;
    }
    // Check bundle-specific permissions.
    if ($account->hasPermission("$operation $bundle revisions") ||
        $account->hasPermission("$operation any $bundle $entity_type_id revisions")) {
      return TRUE;
    }
    // Check entity access.
    $access_operation = $operation === 'revert' ? 'update' : 'delete';
    return $entity->access($access_operation);
  }

  /**
   * Builds operation links for a revision.
   */
  protected function buildOperationLinks(
    ContentEntityInterface $entity,
    ContentEntityInterface $revision,
    int|string $vid,
    bool $has_translations,
    string $langcode,
    bool $revert_permission,
    bool $delete_permission,
  ): array {
    $links = [];
    $entity_type_id = $entity->getEntityTypeId();
    if ($revert_permission) {
      $revert_url = NULL;
      // Try translation revert first.
      if ($has_translations) {
        $revert_url = Url::fromRoute("entity_revision_diff.{$entity_type_id}_revision_revert_translation", [
          $entity_type_id => $entity->id(),
          "{$entity_type_id}_revision" => $vid,
          'langcode' => $langcode,
        ]);
        if (!$revert_url->access()) {
          $revert_url = NULL;
        }
      }
      // Fall back to standard revert.
      if ($revert_url === NULL && $entity->getEntityType()->hasLinkTemplate('revision-revert-form')) {
        $revert_url = Url::fromRoute("entity.{$entity_type_id}.revision_revert_form", [
          $entity_type_id => $entity->id(),
          "{$entity_type_id}_revision" => $vid,
        ]);
      }
      if ($revert_url) {
        $links['revert'] = [
          'title' => $vid < $entity->getRevisionId() ? $this->t('Revert') : $this->t('Set as current revision'),
          'url' => $revert_url,
        ];
      }
    }
    if ($delete_permission && $entity->getEntityType()->hasLinkTemplate('revision-delete-form')) {
      $links['delete'] = [
        'title' => $this->t('Delete'),
        'url' => Url::fromRoute("entity.{$entity_type_id}.revision_delete_form", [
          $entity_type_id => $entity->id(),
          "{$entity_type_id}_revision" => $vid,
        ]),
      ];
    }
    return $links;
  }

  /**
   * Set column attributes and return config array.
   */
  protected function buildSelectColumn(string $name, int|string $return_val, int|string|false $default_val): array {
    return [
      '#type' => 'radio',
      '#title_display' => 'invisible',
      '#name' => $name,
      '#return_value' => $return_val,
      '#default_value' => $default_val,
    ];
  }

  /**
   * Gets all the revision IDs for the given entity.
   */
  protected function getRevisionIds(ContentEntityInterface $entity): array {
    $entity_type = $entity->getEntityType();
    $result = $this->entityTypeManager->getStorage($entity->getEntityTypeId())->getQuery()
      ->condition($entity_type->getKey('id'), $entity->id())
      ->pager($this->config->get('general_settings.revision_pager_limit'))
      ->allRevisions()
      ->sort($entity_type->getKey('revision'), 'DESC')
      ->accessCheck(FALSE)
      ->execute();
    return \array_keys($result);
  }

  /**
   * Set and return configuration for revision.
   */
  protected function buildRevision(Link $link, array $username, ContentEntityInterface $revision, ?ContentEntityInterface $previous_revision = NULL): array {
    return [
      '#type' => 'inline_template',
      '#template' => '{% trans %}{{ date }} by {{ username }}{% endtrans %}{% if message %}<p class="revision-log">{{ message }}</p>{% endif %}',
      '#context' => [
        'date' => $link->toString(),
        'username' => !empty($username) ? $this->renderer->renderInIsolation($username) : '',
        'message' => [
          '#markup' => $this->entityComparison->getRevisionDescription($revision, $previous_revision),
          '#allowed_tags' => Xss::getAdminTagList(),
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $input = $form_state->getUserInput();
    if (!isset($input['radios_left']) || !isset($input['radios_right'])) {
      $form_state->setErrorByName('entity_revisions_table', $this->t('Select two revisions to compare.'));
    }
    elseif ($input['radios_left'] == $input['radios_right']) {
      $form_state->setErrorByName('entity_revisions_table', $this->t('Select different revisions to compare.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $input = $form_state->getUserInput();
    $vid_left = $input['radios_left'];
    $vid_right = $input['radios_right'];
    $entity_id = $input['entity_id'];
    $entity_type_id = $input['entity_type_id'];
    // Always place the older revision on the left side.
    if ($vid_left > $vid_right) {
      $aux = $vid_left;
      $vid_left = $vid_right;
      $vid_right = $aux;
    }
    // Build the redirect URL.
    $redirect_url = Url::fromRoute(
      "entity.{$entity_type_id}.revisions_diff",
      [
        $entity_type_id => $entity_id,
        'left_revision' => $vid_left,
        'right_revision' => $vid_right,
        'filter' => $this->diffLayoutManager->getDefaultLayout(),
      ],
    );
    $form_state->setRedirectUrl($redirect_url);
  }

}
