<?php

declare(strict_types=1);

namespace Drupal\entity_revision_diff\Routing;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Routing\Route;

/**
 * Provides dynamic routes for Group entity when group module is installed.
 */
class GroupRouteProvider implements ContainerInjectionInterface {

  /**
   * Constructs a GroupRouteProvider object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   */
  public function __construct(
    protected ModuleHandlerInterface $moduleHandler,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('module_handler'),
    );
  }

  /**
   * Returns an array of route objects.
   *
   * @return \Symfony\Component\Routing\Route[]
   *   An array of route objects.
   */
  public function routes(): array {
    $routes = [];

    // Only register group routes if the group module is installed.
    if (!$this->moduleHandler->moduleExists('group')) {
      return $routes;
    }

    // Group revision view route.
    $routes['entity.group.revision'] = new Route(
      '/group/{group}/revision/{group_revision}',
      [
        '_controller' => '\Drupal\entity_revision_diff\Controller\EntityRevisionController::showGroupRevision',
        '_title_callback' => '\Drupal\entity_revision_diff\Controller\EntityRevisionController::groupRevisionTitle',
      ],
      [
        '_entity_access' => 'group.view',
        'group' => '\d+',
        'group_revision' => '\d+',
      ],
      [
        '_group_operation_route' => TRUE,
        'parameters' => [
          'group' => [
            'type' => 'entity:group',
          ],
          'group_revision' => [
            'type' => 'entity_revision:group',
          ],
        ],
      ]
    );

    // Group revert translation route.
    $routes['entity_revision_diff.group_revision_revert_translation'] = new Route(
      '/group/{group}/revisions/{group_revision}/revert/{langcode}',
      [
        '_form' => '\Drupal\entity_revision_diff\Form\EntityRevisionRevertTranslationForm',
        '_title' => 'Revert to earlier revision of a translation',
      ],
      [
        '_entity_access' => 'group.update',
        'group' => '\d+',
        'group_revision' => '\d+',
      ],
      [
        '_group_operation_route' => TRUE,
        '_admin_route' => TRUE,
        'parameters' => [
          'group' => [
            'type' => 'entity:group',
          ],
          'group_revision' => [
            'type' => 'entity_revision:group',
          ],
        ],
      ]
    );

    return $routes;
  }

}
