<?php

declare(strict_types=1);

namespace Drupal\Tests\entity_revision_diff\Functional;

use Drupal\Core\File\FileExists;
use Drupal\media\Entity\Media;
use Drupal\media\Entity\MediaType;
use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\Entity\Vocabulary;

/**
 * Tests diff module integration.
 *
 * These tests will fail if:
 * - Diff module changes comparison controller
 * - Diff module changes route structure for revisions_diff
 * - Diff module changes comparison output format
 *
 * @group entity_revision_diff
 */
class DiffIntegrationTest extends EntityRevisionDiffFunctionalTestBase {

  /**
   * Tests diff comparison route exists for block_content.
   *
   * Note: Full diff page test requires complex diff module configuration.
   * Route existence and structure are verified here and in kernel tests.
   */
  public function testBlockContentDiffRouteExists(): void {
    // Verify the route exists via route provider.
    $route_provider = \Drupal::service('router.route_provider');
    $route = $route_provider->getRouteByName('entity.block_content.revisions_diff');
    $this->assertNotNull($route, 'Block content revisions_diff route must exist');
    $this->assertStringContainsString('{left_revision}', $route->getPath());
    $this->assertStringContainsString('{right_revision}', $route->getPath());
    // Verify the route uses diff controller.
    $defaults = $route->getDefaults();
    $this->assertArrayHasKey('_controller', $defaults);
    $this->assertStringContainsString('PluginRevisionController', $defaults['_controller']);
  }

  /**
   * Tests diff comparison route exists for group.
   *
   * Note: Full functional test skipped due to Group module's complex
   * permission system. Route existence is verified in kernel tests.
   */
  public function testGroupDiffRouteExists(): void {
    // Verify the route exists via route provider.
    $route_provider = \Drupal::service('router.route_provider');
    $route = $route_provider->getRouteByName('entity.group.revisions_diff');
    $this->assertNotNull($route, 'Group revisions_diff route must exist');
    $this->assertStringContainsString('{left_revision}', $route->getPath());
    $this->assertStringContainsString('{right_revision}', $route->getPath());
  }

  /**
   * Tests diff comparison page for media.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testMediaDiffComparison(): void {
    $this->loginAsAdmin(['administer media', 'administer media types']);
    // Create media type.
    $media_type = MediaType::create([
      'id' => 'diff_media_test',
      'label' => 'Diff Media Test',
      'source' => 'file',
      'new_revision' => TRUE,
    ]);
    $media_type->save();
    $source_field = $media_type->getSource()->createSourceField($media_type);
    $source_field->getFieldStorageDefinition()->save();
    $source_field->save();
    $media_type
      ->set('source_configuration', [
        'source_field' => $source_field->getName(),
      ])
      ->save();
    // Create a test file.
    $file = \Drupal::service('file.repository')->writeData(
      'test content',
      'public://diff_test.txt',
      FileExists::Replace
    );
    // Create media.
    $media = Media::create([
      'bundle' => 'diff_media_test',
      'name' => 'Initial media name',
      $source_field->getName() => $file->id(),
    ]);
    $media->save();
    $first_vid = $media->getRevisionId();
    // Create second revision.
    $media->setName('Modified media name');
    $media->setNewRevision(TRUE);
    $media->setRevisionLogMessage('Changed name');
    $media->save();
    $second_vid = $media->getRevisionId();
    // Visit diff comparison page.
    $this->drupalGet("/media/{$media->id()}/revisions/view/{$first_vid}/{$second_vid}");
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests diff comparison page for taxonomy_term.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testTaxonomyTermDiffComparison(): void {
    $this->loginAsAdmin(['administer taxonomy']);
    // Create vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'diff_vocab',
      'name' => 'Diff Vocabulary',
    ]);
    $vocabulary->save();
    // Create term.
    $term = Term::create([
      'vid' => 'diff_vocab',
      'name' => 'Original name',
    ]);
    $term->save();
    $first_vid = $term->getRevisionId();
    // Create second revision.
    $term->setName('Modified name');
    $term->setNewRevision(TRUE);
    $term->setRevisionLogMessage('Changed name');
    $term->save();
    $second_vid = $term->getRevisionId();
    // Visit diff comparison page.
    $this->drupalGet("/taxonomy/term/{$term->id()}/revisions/view/{$first_vid}/{$second_vid}");
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests form submission redirects to diff page.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testFormSubmissionRedirectsToDiff(): void {
    $this->loginAsAdmin(['administer taxonomy']);
    // Create vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'redirect_test',
      'name' => 'Redirect Test',
    ]);
    $vocabulary->save();
    // Create term with revisions.
    $term = Term::create([
      'vid' => 'redirect_test',
      'name' => 'Test term',
    ]);
    $term->save();
    $first_vid = $term->getRevisionId();
    $term->setName('Updated');
    $term->setNewRevision(TRUE);
    $term->save();
    $second_vid = $term->getRevisionId();
    // Visit revision overview.
    $this->drupalGet("/taxonomy/term/{$term->id()}/revisions");
    // Submit form with two different revisions.
    $this->submitForm([
      'radios_left' => $first_vid,
      'radios_right' => $second_vid,
    ], 'Compare selected revisions');
    // Should redirect to diff comparison page.
    $this->assertSession()->addressMatches('/revisions\/view/');
  }

  /**
   * Tests that older revision is always on the left.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testOlderRevisionOnLeft(): void {
    $this->loginAsAdmin(['administer taxonomy']);
    // Create vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'order_test',
      'name' => 'Order Test',
    ]);
    $vocabulary->save();
    // Create term with revisions.
    $term = Term::create([
      'vid' => 'order_test',
      'name' => 'Test term',
    ]);
    $term->save();
    $first_vid = $term->getRevisionId();
    $term->setName('Updated');
    $term->setNewRevision(TRUE);
    $term->save();
    $second_vid = $term->getRevisionId();
    // Visit revision overview.
    $this->drupalGet("/taxonomy/term/{$term->id()}/revisions");
    // Submit form with newer revision on left (should be swapped).
    $this->submitForm([
      'radios_left' => $second_vid,
      'radios_right' => $first_vid,
    ], 'Compare selected revisions');
    // URL should have older revision first.
    $current_url = $this->getSession()->getCurrentUrl();
    $this->assertStringContainsString(
      "/{$first_vid}/{$second_vid}",
      $current_url,
      'Older revision should be on the left side of URL'
    );
  }

  /**
   * Tests diff comparison uses correct layout from diff settings.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testDiffLayoutIntegration(): void {
    $this->loginAsAdmin(['administer taxonomy']);
    // Create vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'layout_test',
      'name' => 'Layout Test',
    ]);
    $vocabulary->save();
    // Create term with revisions.
    $term = Term::create([
      'vid' => 'layout_test',
      'name' => 'Test term',
    ]);
    $term->save();
    $first_vid = $term->getRevisionId();
    $term->setName('Updated');
    $term->setNewRevision(TRUE);
    $term->save();
    $second_vid = $term->getRevisionId();
    // Visit diff comparison page with default layout.
    $this->drupalGet("/taxonomy/term/{$term->id()}/revisions/view/{$first_vid}/{$second_vid}");
    $this->assertSession()->statusCodeEquals(200);
    // Page should load without errors (diff module handles the layout).
  }

}
