<?php

declare(strict_types=1);

namespace Drupal\Tests\entity_revision_diff\Functional;

use Drupal\block_content\Entity\BlockContent;
use Drupal\block_content\Entity\BlockContentType;
use Drupal\Core\File\FileExists;
use Drupal\media\Entity\Media;
use Drupal\media\Entity\MediaType;
use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\Entity\Vocabulary;

/**
 * Tests the revision overview form functionality.
 *
 * These tests will fail if:
 * - Core changes form API or table render element
 * - Diff module changes radio button behavior or JavaScript
 * - Entity types change their revision storage API
 *
 * @group entity_revision_diff
 */
class RevisionOverviewFormTest extends EntityRevisionDiffFunctionalTestBase {

  /**
   * Tests revision overview page for block_content.
   *
   * @throws \Behat\Mink\Exception\ElementNotFoundException
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Behat\Mink\Exception\ResponseTextException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testBlockContentRevisionOverview(): void {
    $this->loginAsAdmin(['administer block content']);
    // Create block content type.
    $block_type = BlockContentType::create([
      'id' => 'basic',
      'label' => 'Basic block',
      'revision' => TRUE,
    ]);
    $block_type->save();
    block_content_add_body_field($block_type->id());
    // Create block content with multiple revisions.
    $block = BlockContent::create([
      'type' => 'basic',
      'info' => 'Test block',
      'body' => ['value' => 'Initial content'],
    ]);
    $block->save();
    // Create second revision.
    $block->body->value = 'Updated content';
    $block->setNewRevision(TRUE);
    $block->setRevisionLogMessage('Second revision');
    $block->save();
    // Visit revision overview page.
    $this->drupalGet("/admin/content/block/{$block->id()}/revisions");
    $this->assertSession()->statusCodeEquals(200);
    // Check page title.
    $this->assertSession()->pageTextContains('Revisions for');
    // Check diff elements exist.
    $this->assertSession()->elementExists('css', 'table.diff-revisions');
    $this->assertSession()->elementExists('css', 'input[name="radios_left"]');
    $this->assertSession()->elementExists('css', 'input[name="radios_right"]');
    $this->assertSession()->buttonExists('Compare selected revisions');
    // Check current revision is marked.
    $this->assertSession()->pageTextContains('Current revision');
  }

  /**
   * Tests version_history route exists for group.
   *
   * Note: Full functional test skipped due to Group module's complex
   * permission system. Route and form are verified in kernel tests.
   */
  public function testGroupVersionHistoryRouteExists(): void {
    // Verify the route exists via route provider.
    $route_provider = \Drupal::service('router.route_provider');
    $route = $route_provider->getRouteByName('entity.group.version_history');
    $this->assertNotNull($route, 'Group version_history route must exist');
    // Verify our form is used.
    $defaults = $route->getDefaults();
    $this->assertArrayHasKey('_form', $defaults);
    $this->assertStringContainsString('EntityRevisionOverviewForm', $defaults['_form']);
  }

  /**
   * Tests revision overview page for taxonomy_term.
   *
   * @throws \Behat\Mink\Exception\ElementNotFoundException
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testTaxonomyTermRevisionOverview(): void {
    $this->loginAsAdmin(['administer taxonomy']);
    // Create vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'tags',
      'name' => 'Tags',
    ]);
    $vocabulary->save();
    // Create term with multiple revisions.
    $term = Term::create([
      'vid' => 'tags',
      'name' => 'Test term',
    ]);
    $term->save();
    // Create second revision.
    $term->setName('Updated term');
    $term->setNewRevision(TRUE);
    $term->setRevisionLogMessage('Second revision');
    $term->save();
    // Visit revision overview page.
    $this->drupalGet("/taxonomy/term/{$term->id()}/revisions");
    $this->assertSession()->statusCodeEquals(200);
    // Check diff elements exist.
    $this->assertSession()->elementExists('css', 'table.diff-revisions');
    $this->assertSession()->elementExists('css', 'input[name="radios_left"]');
    $this->assertSession()->elementExists('css', 'input[name="radios_right"]');
  }

  /**
   * Tests revision overview page for media.
   *
   * @throws \Behat\Mink\Exception\ElementNotFoundException
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testMediaRevisionOverview(): void {
    $this->loginAsAdmin(['administer media', 'administer media types']);
    // Create media type.
    $media_type = MediaType::create([
      'id' => 'test_media',
      'label' => 'Test Media',
      'source' => 'file',
      'new_revision' => TRUE,
    ]);
    $media_type->save();
    $source_field = $media_type->getSource()->createSourceField($media_type);
    $source_field->getFieldStorageDefinition()->save();
    $source_field->save();
    $media_type
      ->set('source_configuration', [
        'source_field' => $source_field->getName(),
      ])
      ->save();
    // Create a test file.
    $file = \Drupal::service('file.repository')->writeData(
      'test content',
      'public://test.txt',
      FileExists::Replace
    );
    // Create media with multiple revisions.
    $media = Media::create([
      'bundle' => 'test_media',
      'name' => 'Test media',
      $source_field->getName() => $file->id(),
    ]);
    $media->save();
    // Create second revision.
    $media->setName('Updated media');
    $media->setNewRevision(TRUE);
    $media->setRevisionLogMessage('Second revision');
    $media->save();
    // Visit revision overview page.
    $this->drupalGet("/media/{$media->id()}/revisions");
    $this->assertSession()->statusCodeEquals(200);
    // Check diff elements exist.
    $this->assertSession()->elementExists('css', 'table.diff-revisions');
    $this->assertSession()->elementExists('css', 'input[name="radios_left"]');
    $this->assertSession()->elementExists('css', 'input[name="radios_right"]');
  }

  /**
   * Tests that Compare button is not shown for single revision.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testNoCompareButtonForSingleRevision(): void {
    $this->loginAsAdmin(['administer taxonomy']);
    // Create vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'single_rev',
      'name' => 'Single Rev',
    ]);
    $vocabulary->save();
    // Create term with single revision.
    $term = Term::create([
      'vid' => 'single_rev',
      'name' => 'Single revision term',
    ]);
    $term->save();
    // Visit revision overview page.
    $this->drupalGet("/taxonomy/term/{$term->id()}/revisions");
    $this->assertSession()->statusCodeEquals(200);
    // Compare button should NOT exist for single revision.
    $this->assertSession()->buttonNotExists('Compare selected revisions');
    // Radio buttons should NOT exist.
    $this->assertSession()->elementNotExists('css', 'input[name="radios_left"]');
    $this->assertSession()->elementNotExists('css', 'input[name="radios_right"]');
  }

  /**
   * Tests form validation for same revision selection.
   *
   * @throws \Behat\Mink\Exception\ResponseTextException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testFormValidationSameRevision(): void {
    $this->loginAsAdmin(['administer taxonomy']);
    // Create vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'validation_test',
      'name' => 'Validation Test',
    ]);
    $vocabulary->save();
    // Create term with multiple revisions.
    $term = Term::create([
      'vid' => 'validation_test',
      'name' => 'Test term',
    ]);
    $term->save();
    $term->setName('Updated term');
    $term->setNewRevision(TRUE);
    $term->save();
    // Visit revision overview and try to compare same revision.
    $this->drupalGet("/taxonomy/term/{$term->id()}/revisions");
    // Submit form with same revision selected (simulate via POST).
    $vid = $term->getRevisionId();
    $this->submitForm([
      'radios_left' => $vid,
      'radios_right' => $vid,
    ], 'Compare selected revisions');
    // Should see error message.
    $this->assertSession()->pageTextContains('Select different revisions to compare');
  }

  /**
   * Tests diff library is attached to the form.
   *
   * @throws \Behat\Mink\Exception\ElementNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testDiffLibraryAttached(): void {
    $this->loginAsAdmin(['administer taxonomy']);
    // Create vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'library_test',
      'name' => 'Library Test',
    ]);
    $vocabulary->save();
    // Create term with multiple revisions.
    $term = Term::create([
      'vid' => 'library_test',
      'name' => 'Test term',
    ]);
    $term->save();
    $term->setName('Updated');
    $term->setNewRevision(TRUE);
    $term->save();
    $this->drupalGet("/taxonomy/term/{$term->id()}/revisions");
    // Check that diff library is attached (by checking CSS class).
    $this->assertSession()->elementExists('css', 'table.diff-revisions');
  }

}
