<?php

declare(strict_types=1);

namespace Drupal\Tests\entity_revision_diff\Functional;

use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\Entity\Vocabulary;

/**
 * Tests the translation revert form functionality.
 *
 * These tests will fail if:
 * - Core changes translation API
 * - Core changes revision API for translations
 * - Core changes form confirm API
 *
 * @group entity_revision_diff
 */
class TranslationRevertFormTest extends EntityRevisionDiffFunctionalTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'block',
    'block_content',
    'media',
    'taxonomy',
    'field',
    'text',
    'filter',
    'file',
    'image',
    'user',
    'entity',
    'flexible_permissions',
    'group',
    'options',
    'variationcache',
    'diff',
    'entity_revision_diff',
    'language',
    'content_translation',
  ];

  /**
   * {@inheritdoc}
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function setUp(): void {
    parent::setUp();
    // Create and enable German language.
    ConfigurableLanguage::createFromLangcode('de')->save();
  }

  /**
   * Tests translation revert form access.
   *
   * @throws \Behat\Mink\Exception\ElementNotFoundException
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Behat\Mink\Exception\ResponseTextException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testTranslationRevertFormAccess(): void {
    $this->loginAsAdmin([
      'administer taxonomy',
      'translate any entity',
      'create content translations',
    ]);
    // Create vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'trans_vocab',
      'name' => 'Trans Vocabulary',
    ]);
    $vocabulary->save();
    // Enable translation.
    \Drupal::service('content_translation.manager')
      ->setEnabled('taxonomy_term', 'trans_vocab', TRUE);
    // Create term with translation.
    $term = Term::create([
      'vid' => 'trans_vocab',
      'name' => 'English name',
      'langcode' => 'en',
    ]);
    $term->save();
    $first_vid = $term->getRevisionId();
    // Add German translation.
    $term->addTranslation('de', ['name' => 'German name']);
    $term->setNewRevision(TRUE);
    $term->save();
    // Visit translation revert form.
    $this->drupalGet("/taxonomy/term/{$term->id()}/revisions/{$first_vid}/revert/en");
    $this->assertSession()->statusCodeEquals(200);
    // Check form elements.
    $this->assertSession()->pageTextContains('Are you sure you want to revert');
    $this->assertSession()->buttonExists('Revert');
  }

  /**
   * Tests translation revert form contains shared content checkbox.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testTranslationRevertFormHasSharedContentCheckbox(): void {
    $this->loginAsAdmin([
      'administer taxonomy',
      'translate any entity',
      'create content translations',
    ]);
    // Create vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'shared_test',
      'name' => 'Shared Test',
    ]);
    $vocabulary->save();
    // Enable translation.
    \Drupal::service('content_translation.manager')
      ->setEnabled('taxonomy_term', 'shared_test', TRUE);
    // Create term with translation and revisions.
    $term = Term::create([
      'vid' => 'shared_test',
      'name' => 'English name',
      'langcode' => 'en',
    ]);
    $term->save();
    $first_vid = $term->getRevisionId();
    $term->addTranslation('de', ['name' => 'German name']);
    $term->setNewRevision(TRUE);
    $term->save();
    // Visit translation revert form.
    $this->drupalGet("/taxonomy/term/{$term->id()}/revisions/{$first_vid}/revert/en");
    // Should see the shared content checkbox (or confirmation message).
    // The checkbox only appears when untranslatable fields are NOT configured
    // to affect only default translation.
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests cancel URL redirects to version history.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testCancelRedirectsToVersionHistory(): void {
    $this->loginAsAdmin([
      'administer taxonomy',
      'translate any entity',
    ]);
    // Create vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'cancel_test',
      'name' => 'Cancel Test',
    ]);
    $vocabulary->save();
    \Drupal::service('content_translation.manager')
      ->setEnabled('taxonomy_term', 'cancel_test', TRUE);
    $term = Term::create([
      'vid' => 'cancel_test',
      'name' => 'Test term',
      'langcode' => 'en',
    ]);
    $term->save();
    $vid = $term->getRevisionId();
    $term->setName('Updated');
    $term->setNewRevision(TRUE);
    $term->save();
    // Visit translation revert form.
    $this->drupalGet("/taxonomy/term/{$term->id()}/revisions/{$vid}/revert/en");
    // Find cancel link.
    $this->assertSession()->linkExists('Cancel');
    // Click cancel.
    $this->clickLink('Cancel');
    // Should be back at version history.
    $this->assertSession()->addressEquals("/taxonomy/term/{$term->id()}/revisions");
  }

  /**
   * Tests successful translation revert.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Behat\Mink\Exception\ResponseTextException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testSuccessfulTranslationRevert(): void {
    $this->loginAsAdmin([
      'administer taxonomy',
      'translate any entity',
      'create content translations',
    ]);
    // Create vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'revert_success',
      'name' => 'Revert Success',
    ]);
    $vocabulary->save();
    \Drupal::service('content_translation.manager')
      ->setEnabled('taxonomy_term', 'revert_success', TRUE);
    // Create term.
    $term = Term::create([
      'vid' => 'revert_success',
      'name' => 'Original name',
      'langcode' => 'en',
    ]);
    $term->save();
    $first_vid = $term->getRevisionId();
    // Create new revision with different name.
    $term->setName('Modified name');
    $term->setNewRevision(TRUE);
    $term->save();
    // Visit translation revert form for first revision.
    $this->drupalGet("/taxonomy/term/{$term->id()}/revisions/{$first_vid}/revert/en");
    // Submit revert form.
    $this->submitForm([], 'Revert');
    // Should see success message.
    $this->assertSession()->pageTextContains('has been reverted');
    // Should be redirected to version history.
    $this->assertSession()->addressMatches('/revisions/');
  }

}
