<?php

declare(strict_types=1);

namespace Drupal\Tests\entity_revision_diff\Kernel;

use Drupal\diff\Routing\DiffRouteProvider;

/**
 * Tests entity type alterations made by entity_revision_diff module.
 *
 * These tests will fail if:
 * - Diff module changes DiffRouteProvider class location or interface
 * - Core changes entity type handler registration
 * - Core changes link template API
 *
 * @group entity_revision_diff
 */
class EntityTypeAlterTest extends EntityRevisionDiffKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'filter',
    'node',
    'diff',
    'block',
    'block_content',
    'media',
    'image',
    'file',
    'taxonomy',
    'entity',
    'flexible_permissions',
    'group',
    'options',
    'variationcache',
    'entity_revision_diff',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('block_content');
    $this->installEntitySchema('media');
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('taxonomy_vocabulary');
    $this->installEntitySchema('group');
    $this->installEntitySchema('group_content');
    $this->installEntitySchema('group_config_wrapper');
    $this->installEntitySchema('file');
    $this->installConfig(['block_content', 'media', 'taxonomy', 'group']);
  }

  /**
   * Tests that DiffRouteProvider class exists and is accessible.
   *
   * This test will fail if diff module changes the class location.
   */
  public function testDiffRouteProviderClassExists(): void {
    $this->assertTrue(
      class_exists(DiffRouteProvider::class),
      'DiffRouteProvider class must exist at Drupal\diff\Routing\DiffRouteProvider'
    );
  }

  /**
   * Tests that DiffRouteProvider is added to supported entity types.
   *
   * @dataProvider supportedEntityTypesProvider
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function testDiffRouteProviderAdded(string $entity_type_id): void {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    $this->assertNotNull(
      $entity_type,
      "Entity type '$entity_type_id' must be defined"
    );
    // Check that the entity type is revisionable.
    $this->assertTrue(
      $entity_type->isRevisionable(),
      "Entity type '$entity_type_id' must be revisionable"
    );
    // Check that DiffRouteProvider is registered.
    $handlers = $entity_type->get('handlers');
    $this->assertArrayHasKey(
      'route_provider',
      $handlers,
      "Entity type '$entity_type_id' must have route_provider handlers"
    );
    $this->assertArrayHasKey(
      'diff',
      $handlers['route_provider'],
      "Entity type '$entity_type_id' must have 'diff' route provider"
    );
    $this->assertEquals(
      DiffRouteProvider::class,
      $handlers['route_provider']['diff'],
      "Entity type '$entity_type_id' diff route provider must be DiffRouteProvider class"
    );
  }

  /**
   * Tests that revisions-diff link template is added to supported entity types.
   *
   * @dataProvider supportedEntityTypesProvider
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function testRevisionsDiffLinkTemplateAdded(string $entity_type_id): void {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    $this->assertTrue(
      $entity_type->hasLinkTemplate('revisions-diff'),
      "Entity type '$entity_type_id' must have 'revisions-diff' link template"
    );
    $link_template = $entity_type->getLinkTemplate('revisions-diff');
    $this->assertNotEmpty(
      $link_template,
      "Entity type '$entity_type_id' revisions-diff link template must not be empty"
    );
    // Verify the link template contains required placeholders.
    $this->assertStringContainsString(
      '{left_revision}',
      $link_template,
      "Link template must contain {left_revision} placeholder"
    );
    $this->assertStringContainsString(
      '{right_revision}',
      $link_template,
      "Link template must contain {right_revision} placeholder"
    );
  }

  /**
   * Tests that node entity type is NOT altered (handled by diff module).
   */
  public function testNodeEntityTypeNotAltered(): void {
    // Node should be handled by diff module directly, not our module.
    $supported_types = entity_revision_diff_supported_entity_types();
    $this->assertArrayNotHasKey(
      'node',
      $supported_types,
      'Node entity type should NOT be in our supported types (handled by diff module)'
    );
  }

  /**
   * Tests entity type definitions match expected structure.
   *
   * This detects if core changes entity type definition API.
   *
   * @dataProvider supportedEntityTypesProvider
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function testEntityTypeDefinitionStructure(string $entity_type_id): void {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    // Test core API methods exist and work.
    $this->assertIsString($entity_type->id());
    $this->assertNotEmpty($entity_type->getKey('id'));
    $this->assertNotEmpty($entity_type->getKey('revision'));
    $this->assertTrue(method_exists($entity_type, 'isRevisionable'));
    $this->assertTrue(method_exists($entity_type, 'hasLinkTemplate'));
    $this->assertTrue(method_exists($entity_type, 'getLinkTemplate'));
    $this->assertTrue(method_exists($entity_type, 'get'));
    $this->assertTrue(method_exists($entity_type, 'set'));
  }

  /**
   * Data provider for supported entity types.
   *
   * @return array
   *   Test data.
   */
  public static function supportedEntityTypesProvider(): array {
    return [
      'group' => ['group'],
      'block_content' => ['block_content'],
      'media' => ['media'],
      'taxonomy_term' => ['taxonomy_term'],
    ];
  }

}
