<?php

declare(strict_types=1);

namespace Drupal\Tests\entity_revision_diff\Kernel;

use Drupal\entity_revision_diff\EntityDiffPermissions;

/**
 * Tests permissions provided by entity_revision_diff module.
 *
 * These tests will fail if:
 * - Core changes permission handler API
 * - Core changes entity type bundle info API
 * - Module changes permission naming convention
 *
 * @group entity_revision_diff
 */
class PermissionsTest extends EntityRevisionDiffKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'filter',
    'node',
    'diff',
    'block',
    'block_content',
    'media',
    'image',
    'file',
    'taxonomy',
    'entity',
    'flexible_permissions',
    'group',
    'options',
    'variationcache',
    'entity_revision_diff',
  ];

  /**
   * The permission handler.
   *
   * @var \Drupal\user\PermissionHandlerInterface
   */
  protected $permissionHandler;

  /**
   * {@inheritdoc}
   * @throws \Exception
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('block_content');
    $this->installEntitySchema('media');
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('taxonomy_vocabulary');
    $this->installEntitySchema('group');
    $this->installEntitySchema('group_content');
    $this->installEntitySchema('group_config_wrapper');
    $this->installEntitySchema('file');
    $this->installConfig(['block_content', 'media', 'taxonomy', 'group']);
    $this->permissionHandler = $this->container->get('user.permissions');
  }

  /**
   * Tests that global permissions exist for all supported entity types.
   *
   * @dataProvider globalPermissionsProvider
   */
  public function testGlobalPermissionsExist(string $permission): void {
    $permissions = $this->permissionHandler->getPermissions();
    $this->assertArrayHasKey(
      $permission,
      $permissions,
      "Permission '$permission' must exist"
    );
  }

  /**
   * Tests that EntityDiffPermissions class exists and is instantiable.
   */
  public function testEntityDiffPermissionsClassExists(): void {
    $this->assertTrue(
      class_exists(EntityDiffPermissions::class),
      'EntityDiffPermissions class must exist'
    );
    $permissions_class = EntityDiffPermissions::create($this->container);
    $this->assertInstanceOf(
      EntityDiffPermissions::class,
      $permissions_class
    );
  }

  /**
   * Tests that bundlePermissions method returns expected structure.
   */
  public function testBundlePermissionsStructure(): void {
    $permissions_class = EntityDiffPermissions::create($this->container);
    $permissions = $permissions_class->bundlePermissions();
    $this->assertIsArray($permissions);
    // Each permission should have title and description.
    foreach ($permissions as $permission_name => $permission_info) {
      $this->assertIsString($permission_name);
      $this->assertArrayHasKey(
        'title',
        $permission_info,
        "Permission '$permission_name' must have 'title'"
      );
      $this->assertArrayHasKey(
        'description',
        $permission_info,
        "Permission '$permission_name' must have 'description'"
      );
    }
  }

  /**
   * Tests permission naming convention.
   */
  public function testPermissionNamingConvention(): void {
    $permissions_class = EntityDiffPermissions::create($this->container);
    $permissions = $permissions_class->bundlePermissions();
    foreach (array_keys($permissions) as $permission_name) {
      // Permissions should match one of these patterns:
      // - Bundle: "action bundle_id revisions"
      // - Global: "action all entity_type_id revisions".
      $this->assertMatchesRegularExpression(
        '/^(view|revert|delete) (all )?\S+ revisions$/',
        $permission_name,
        "Permission '$permission_name' must match naming convention"
      );
    }
  }

  /**
   * Tests that permissions are generated for bundles when they exist.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testBundlePermissionsGenerated(): void {
    // Create a group type to have at least one bundle.
    $group_type_storage = $this->entityTypeManager->getStorage('group_type');
    $group_type = $group_type_storage->create([
      'id' => 'test_group_type',
      'label' => 'Test Group Type',
      'creator_wizard' => FALSE,
    ]);
    $group_type->save();
    // Rebuild permissions.
    $permissions_class = EntityDiffPermissions::create($this->container);
    $permissions = $permissions_class->bundlePermissions();
    // Should have permissions for test_group_type bundle.
    $this->assertArrayHasKey(
      'view test_group_type revisions',
      $permissions
    );
    $this->assertArrayHasKey(
      'revert test_group_type revisions',
      $permissions
    );
    $this->assertArrayHasKey(
      'delete test_group_type revisions',
      $permissions
    );
  }

  /**
   * Tests that permissions are not generated for non-revisionable entity types.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function testNoPermissionsForNonRevisionableEntities(): void {
    // The supported_entity_types function should only return revisionable types.
    $supported_types = entity_revision_diff_supported_entity_types();

    foreach (array_keys($supported_types) as $entity_type_id) {
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      $this->assertTrue(
        $entity_type->isRevisionable(),
        "Supported entity type '$entity_type_id' must be revisionable"
      );
    }
  }

  /**
   * Data provider for global permissions.
   *
   * @return array
   *   Test data.
   */
  public static function globalPermissionsProvider(): array {
    return [
      // Group permissions.
      ['view all group revisions'],
      ['revert all group revisions'],
      ['delete all group revisions'],
      // Block content permissions.
      ['view all block_content revisions'],
      ['revert all block_content revisions'],
      ['delete all block_content revisions'],
      // Media permissions.
      ['view all media revisions'],
      ['revert all media revisions'],
      ['delete all media revisions'],
      // Taxonomy term permissions.
      ['view all taxonomy_term revisions'],
      ['revert all taxonomy_term revisions'],
      ['delete all taxonomy_term revisions'],
    ];
  }

}
