<?php

declare(strict_types=1);

namespace Drupal\Tests\entity_revision_diff\Kernel;

use Drupal\entity_revision_diff\Form\EntityRevisionOverviewForm;
use Symfony\Component\Routing\Exception\RouteNotFoundException;

/**
 * Tests route alterations made by entity_revision_diff module.
 *
 * These tests will fail if:
 * - Core changes route naming conventions for entity version_history
 * - Diff module changes revisions_diff route naming
 * - Group/Media/Block/Taxonomy modules change their revision route structure
 *
 * @group entity_revision_diff
 */
class RouteAlterTest extends EntityRevisionDiffKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'filter',
    'node',
    'diff',
    'block',
    'block_content',
    'media',
    'image',
    'file',
    'taxonomy',
    'entity',
    'flexible_permissions',
    'group',
    'options',
    'variationcache',
    'entity_revision_diff',
  ];

  /**
   * The route provider service.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected $routeProvider;

  /**
   * {@inheritdoc}
   * @throws \Exception
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('block_content');
    $this->installEntitySchema('media');
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('taxonomy_vocabulary');
    $this->installEntitySchema('group');
    $this->installEntitySchema('group_content');
    $this->installEntitySchema('group_config_wrapper');
    $this->installEntitySchema('file');
    $this->installConfig(['block_content', 'media', 'taxonomy', 'group']);
    // Rebuild routes to ensure our alterations are applied.
    $this->container->get('router.builder')->rebuild();
    $this->routeProvider = $this->container->get('router.route_provider');
  }

  /**
   * Tests that version_history routes exist for all supported entity types.
   *
   * @dataProvider supportedEntityTypesProvider
   */
  public function testVersionHistoryRouteExists(string $entity_type_id): void {
    $route_name = "entity.{$entity_type_id}.version_history";
    try {
      $route = $this->routeProvider->getRouteByName($route_name);
      $this->assertNotNull($route, "Route '$route_name' must exist");
    }
    catch (RouteNotFoundException $e) {
      $this->fail("Route '$route_name' must exist but was not found");
    }
  }

  /**
   * Tests that version_history routes use our form.
   *
   * @dataProvider supportedEntityTypesProvider
   */
  public function testVersionHistoryRouteUsesOurForm(string $entity_type_id): void {
    $route_name = "entity.{$entity_type_id}.version_history";
    $route = $this->routeProvider->getRouteByName($route_name);
    $defaults = $route->getDefaults();
    $this->assertArrayHasKey(
      '_form',
      $defaults,
      "Route '$route_name' must have _form default"
    );
    $this->assertEquals(
      '\\' . EntityRevisionOverviewForm::class,
      $defaults['_form'],
      "Route '$route_name' must use EntityRevisionOverviewForm"
    );
  }

  /**
   * Tests that version_history routes have entity_type_id option.
   *
   * @dataProvider supportedEntityTypesProvider
   */
  public function testVersionHistoryRouteHasEntityTypeOption(string $entity_type_id): void {
    $route_name = "entity.{$entity_type_id}.version_history";
    $route = $this->routeProvider->getRouteByName($route_name);
    $entity_type_option = $route->getOption('entity_type_id');
    $this->assertEquals(
      $entity_type_id,
      $entity_type_option,
      "Route '$route_name' must have entity_type_id option set to '$entity_type_id'"
    );
  }

  /**
   * Tests that revisions_diff routes exist (created by DiffRouteProvider).
   *
   * @dataProvider supportedEntityTypesProvider
   */
  public function testRevisionsDiffRouteExists(string $entity_type_id): void {
    $route_name = "entity.{$entity_type_id}.revisions_diff";
    try {
      $route = $this->routeProvider->getRouteByName($route_name);
      $this->assertNotNull($route, "Route '$route_name' must exist");
    }
    catch (RouteNotFoundException $e) {
      $this->fail("Route '$route_name' must exist but was not found (DiffRouteProvider may have changed)");
    }
  }

  /**
   * Tests that revisions_diff routes have correct path structure.
   *
   * @dataProvider supportedEntityTypesProvider
   */
  public function testRevisionsDiffRoutePathStructure(string $entity_type_id): void {
    $route_name = "entity.{$entity_type_id}.revisions_diff";
    $route = $this->routeProvider->getRouteByName($route_name);
    $path = $route->getPath();
    // Verify path contains required parameters.
    $this->assertStringContainsString(
      '{left_revision}',
      $path,
      "Route path must contain {left_revision} parameter"
    );
    $this->assertStringContainsString(
      '{right_revision}',
      $path,
      "Route path must contain {right_revision} parameter"
    );
  }

  /**
   * Tests that our translation revert routes exist.
   *
   * @dataProvider supportedEntityTypesProvider
   */
  public function testTranslationRevertRouteExists(string $entity_type_id): void {
    $route_name = "entity_revision_diff.{$entity_type_id}_revision_revert_translation";
    try {
      $route = $this->routeProvider->getRouteByName($route_name);
      $this->assertNotNull($route, "Route '$route_name' must exist");
    }
    catch (RouteNotFoundException $e) {
      $this->fail("Route '$route_name' must exist but was not found");
    }
  }

  /**
   * Tests translation revert route structure.
   *
   * @dataProvider supportedEntityTypesProvider
   */
  public function testTranslationRevertRouteStructure(string $entity_type_id): void {
    $route_name = "entity_revision_diff.{$entity_type_id}_revision_revert_translation";
    $route = $this->routeProvider->getRouteByName($route_name);
    // Check path contains langcode parameter.
    $path = $route->getPath();
    $this->assertStringContainsString(
      '{langcode}',
      $path,
      "Translation revert route must contain {langcode} parameter"
    );
    // Check form is set.
    $defaults = $route->getDefaults();
    $this->assertArrayHasKey('_form', $defaults);
    $this->assertStringContainsString(
      'EntityRevisionRevertTranslationForm',
      $defaults['_form']
    );
    // Check requirements.
    $requirements = $route->getRequirements();
    $this->assertArrayHasKey(
      '_entity_access',
      $requirements,
      "Route must have _entity_access requirement"
    );
  }

  /**
   * Tests that group route has special options.
   */
  public function testGroupRouteHasSpecialOptions(): void {
    $route = $this->routeProvider->getRouteByName('entity.group.version_history');
    $this->assertTrue(
      $route->getOption('_group_operation_route'),
      "Group version_history route must have _group_operation_route option"
    );
  }

  /**
   * Data provider for supported entity types.
   *
   * @return array
   *   Test data.
   */
  public static function supportedEntityTypesProvider(): array {
    return [
      'group' => ['group'],
      'block_content' => ['block_content'],
      'media' => ['media'],
      'taxonomy_term' => ['taxonomy_term'],
    ];
  }

}
