<?php

declare(strict_types=1);

namespace Drupal\entity_share_client\Plugin\EntityShareClient\Processor;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\entity_share_client\Attribute\ImportProcessor;
use Drupal\entity_share_client\ImportProcessor\ImportProcessorPluginBase;
use Drupal\entity_share_client\RuntimeImportContext;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Sets the revision author if the user exists on the client.
 *
 * @todo Consider adding options for what to set the author to, out of anonymous
 * / current user / local copy of remote author.
 */
#[ImportProcessor(
  id: 'revision_author',
  label: new TranslatableMarkup('Revision author (experimental)'),
  description: new TranslatableMarkup("Sets the revision author if the user exists on the client. Must go after the 'revision' processor and before the 'entity_reference' processor."),
  stages: [
    // Needs to go after the revision processor but before the entity_reference,
    // because EntityReference::processEntity() does a save, which will reset
    // the new revision flag on the entity.
    'process_entity' => 20,
  ],
)]
class RevisionAuthor extends ImportProcessorPluginBase {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity.repository'),
    );
  }

  /**
   * Creates a RevisionAuthor instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entityRepository
   *   The entity repository service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected EntityRepositoryInterface $entityRepository,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function processEntity(RuntimeImportContext $runtime_import_context, ContentEntityInterface $processed_entity, array $entity_json_data) {
    // Don't do anything for an entity that is not revisionable.
    if (!$processed_entity->getEntityType()->isRevisionable()) {
      return;
    }

    $entity_type = $processed_entity->getEntityType();

    // Don't do anything for an entity type that has no revision user field.
    if (!$entity_type->hasRevisionMetadataKey('revision_user')) {
      return;
    }

    // Don't do anything if the processed entity is not being saved as a new
    // revision.
    if (!$processed_entity->isNewRevision()) {
      return;
    }

    $revision_user_field_name = $entity_type->getRevisionMetadataKey('revision_user');

    $field_mappings = $runtime_import_context->getFieldMappings();
    $entity_type_id = $processed_entity->getEntityTypeId();
    $entity_bundle = $processed_entity->bundle();

    $revision_user_public_name = $field_mappings[$entity_type_id][$entity_bundle][$revision_user_field_name] ?? NULL;

    // Don't do anything if there's no revision user data.
    if (empty($revision_user_public_name)) {
      return;
    }
    if (!isset($entity_json_data['relationships'][$revision_user_public_name]['data']['id'])) {
      return;
    }

    $revision_user_uuid = $entity_json_data['relationships'][$revision_user_public_name]['data']['id'];

    $local_user = $this->entityRepository->loadEntityByUuid('user', $revision_user_uuid);

    // Don't do anything if there's no local version of the revision author.
    if (empty($local_user)) {
      return;
    }

    $processed_entity->setRevisionUserId($local_user->id());
  }

}
