<?php

namespace Drupal\entity_share_auto_server\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\State\StateInterface;
use Drupal\entity_share_auto_server\Service\EntityShareAutoServerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class EntityShareAutoServerConfig definition.
 */
class EntityShareAutoServerConfig extends ConfigFormBase {

  /**
   * State service.
   *
   * @var \Drupal\Core\State\State
   */
  protected $state;

  /**
   * Module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a \Drupal\system\ConfigFormBase object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\State\StateInterface $state
   *   State service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   Module handler service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, StateInterface $state, ModuleHandlerInterface $module_handler) {
    $this->state = $state;
    $this->moduleHandler = $module_handler;
    parent::__construct($config_factory);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('state'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'entity_share_auto.entity_share_auto_server.config',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_share_auto_server_config';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('entity_share_auto.entity_share_auto_server.config');
    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API key'),
      '#description' => $this->t('API key used when server communicate with client. Make sure to add same API key on entity share auto client configuration. Leave empty to make open API calls.'),
      '#maxlength' => 128,
      '#size' => 64,
      '#default_value' => $config->get('api_key'),
    ];
    $form['published'] = [
      '#type' => 'checkbox',
      '#title' => $this
        ->t('Export published only'),
      '#default_value' => $config->get('published'),
      '#description' => $this->t("Check 'Export published only' if only published content needs to be export."),
    ];
    $form['entity_share_auto'] = [
      '#type' => 'container',
      '#title' => $this
        ->t('Basic Auth'),
      '#states' => [
        'invisible' => [
          'input[name="published"]' => [
            'checked' => TRUE,
          ],
        ],
      ],
      '#description' => $this->t("Provide username and password of user having access to unpublished content."),
    ];
    $credentials = $this->state->get(EntityShareAutoServerInterface::BASIC_AUTH_KEY_VALUE_COLLECTION);
    $form['entity_share_auto']['username'] = [
      '#type' => 'textfield',
      '#required' => FALSE,
      '#title' => $this->t('Username'),
      '#default_value' => $credentials['username'] ?? '',
    ];

    $form['entity_share_auto']['password'] = [
      '#type' => 'password',
      '#required' => FALSE,
      '#title' => $this->t('Password'),
      '#default_value' => $credentials['password'] ?? '',
    ];

    if ($this->moduleHandler->moduleExists('entity_share_auto_instant')) {
      $form['instant_export'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Export instantly'),
        '#default_value' => $config->get('instant_export'),
        '#description' => $this->t("Export entities instantly without queuing in export and import queue. On client this will start importing entity instantly."),
      ];
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if ($form_state->getValue('username')
      && $form_state->getValue('password')
      && !$form_state->getValue('published')
      && !$this->moduleHandler->moduleExists('basic_auth')
    ) {
      $form_state->setErrorByName('username', $this->t("'HTTP Basic Authentication' module is not enabled."));
    }

    // @todo Change the autogenerated stub.
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $config = $this->config('entity_share_auto.entity_share_auto_server.config')
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('published', $form_state->getValue('published'));
    if ($this->moduleHandler->moduleExists('entity_share_auto_instant')) {
      $config->set('instant_export', $form_state->getValue('instant_export'));
    }
    $config->save();

    if ($form_state->getValue('username') && $form_state->getValue('password') && !$form_state->getValue('published')) {

      $username = $form_state->getValue('username');
      $password = $form_state->getValue('password');
      $this->state->set(EntityShareAutoServerInterface::BASIC_AUTH_KEY_VALUE_COLLECTION, [
        'username' => $username,
        'password' => $password,
      ]);
    }
    else {
      $this->state->set(EntityShareAutoServerInterface::BASIC_AUTH_KEY_VALUE_COLLECTION, []);
    }
  }

}
