<?php

namespace Drupal\entity_share_trash_ignore\Plugin\EntityShareClient\Processor;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\entity_share_client\ImportProcessor\ImportProcessorPluginBase;
use Drupal\entity_share_client\RuntimeImportContext;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Set status to unpublished or keep it after import.
 *
 * @ImportProcessor(
 *   id = "trash_ignore",
 *   label = @Translation("Ignore trash"),
 *   description = @Translation("Ignore entities that are in the trash when
 *   doing an import. Ignored entities are never imported"),
 *   stages = {
 *     "is_entity_importable" = -5,
 *   },
 *   locked = true,
 * )
 */
class IgnoreTrash extends ImportProcessorPluginBase implements PluginFormInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity type bundle info.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfo
   */
  protected $entityBundleInfo;

  /**
   * The resource type repository.
   *
   * @var \Drupal\jsonapi\ResourceType\ResourceTypeRepositoryInterface
   */
  protected $resourceTypeRepository;

  /**
   * The Entity import state information service.
   *
   * @var \Drupal\entity_share_client\Service\StateInformationInterface
   */
  protected $stateInformation;

  /**
   * A logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The trash manager.
   *
   * @var \Drupal\trash\TrashManagerInterface
   */
  protected $trashManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityBundleInfo = $container->get('entity_type.bundle.info');
    $instance->resourceTypeRepository = $container->get('jsonapi.resource_type.repository');
    $instance->stateInformation = $container->get('entity_share_client.state_information');
    $instance->logger = $container->get('logger.factory')->get('piipe_entity_share');
    $instance->trashManager = $container->get('trash.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function isEntityImportable(RuntimeImportContext $runtime_import_context, array $entity_json_data) {
    [$type, $bundle] = explode('--', $entity_json_data['type'], 2);
    if (!$this->trashManager->isEntityTypeEnabled($type, $bundle)) {
      return TRUE;
    }
    $storage = $this->entityTypeManager->getStorage($type);
    $entity = $this->trashManager->executeInTrashContext('inactive', function () use ($storage, $entity_json_data) {
      return $storage->loadByProperties([
        'uuid' => $entity_json_data['id'],
      ]);
    });
    if (empty($entity)) {
      return TRUE;
    }
    $entity = reset($entity);
    if ($entity->get('deleted')->isEmpty()) {
      return TRUE;
    }
    $field_mappings = $runtime_import_context->getFieldMappings();
    $entity_keys = $storage->getEntityType()->getKeys();
    $label_public_name = $field_mappings[$type][$bundle][$entity_keys['label']];
    $entity_label = $entity_json_data['id'];
    if (!empty($entity_json_data['attributes'][$label_public_name])) {
      $entity_label = $entity_json_data['attributes'][$label_public_name];
    }
    $this->messenger()->addError($this->t('Trying to import a entity that is in trash (@e).', ['@e' => $entity_label]));
    $this->logger->warning('Entity that is in trash is not imported: "@e"', ['@e' => $entity_label]);
    return FALSE;
  }

}
