<?php

namespace Drupal\entity_splitter;

/**
 * Value object representing the result of a migration run.
 */
class EntitySplitterResult {

  /**
   * The plugin id.
   *
   * @var string
   */
  protected string $pluginId;

  /**
   * How many entities were processed.
   *
   * @var int
   */
  private int $processed;

  /**
   * How many entities were created.
   *
   * @var int
   */
  private int $created;

  /**
   * How many entities were updated.
   *
   * @var int
   */
  private int $updated;

  /**
   * How many entities failed to process.
   *
   * @var int
   */
  private int $failed;

  /**
   * The messages.
   *
   * @var array|mixed
   */
  private mixed $messages;

  /**
   * Whether this is a dry run.
   *
   * @var bool
   */
  private bool $dryRun;

  /**
   * Constructs an EntitySplitterResult object.
   *
   * @param string $pluginId
   *   The plugin id.
   * @param int $processed
   *   The processed variable.
   * @param int $created
   *   The created variable.
   * @param int $updated
   *   The updated variable.
   * @param int $failed
   *   The failed variable.
   * @param mixed $messages
   *   The messages variable.
   * @param bool $dryRun
   *   The dry run variable.
   */
  public function __construct(
    string $pluginId,
    int $processed = 0,
    int $created = 0,
    int $updated = 0,
    int $failed = 0,
    $messages = [],
    bool $dryRun = FALSE,
  ) {
    $this->pluginId = $pluginId;
    $this->processed = $processed;
    $this->created = $created;
    $this->updated = $updated;
    $this->failed = $failed;
    $this->messages = $messages;
    $this->dryRun = $dryRun;
  }

  /**
   * Returns a printable summary string.
   *
   * @return string
   *   The summary.
   */
  public function getSummary(): string {
    return sprintf(
      'Plugin %s: processed=%d, created=%d, updated=%d, failed=%d%s',
      $this->pluginId,
      $this->processed,
      $this->created,
      $this->updated,
      $this->failed,
      $this->dryRun ? ' (dry-run)' : ''
    );
  }

  /**
   * Converts the result to an array.
   *
   * @return array
   *   Returns the result as an array.
   */
  public function toArray(): array {
    return [
      'plugin' => $this->pluginId,
      'processed' => $this->processed,
      'created' => $this->created,
      'updated' => $this->updated,
      'failed' => $this->failed,
      'messages' => $this->messages,
      'dry_run' => $this->dryRun,
    ];
  }

  /**
   * Gets the plugin id.
   */
  public function getPluginId(): string {
    return $this->pluginId;
  }

  /**
   * Gets processed count.
   */
  public function getProcessed(): int {
    return $this->processed;
  }

  /**
   * Gets created count.
   */
  public function getCreated(): int {
    return $this->created;
  }

  /**
   * Gets updated count.
   */
  public function getUpdated(): int {
    return $this->updated;
  }

  /**
   * Gets failed count.
   */
  public function getFailed(): int {
    return $this->failed;
  }

  /**
   * Gets messages.
   *
   * @return string
   *   The messages as a string.
   */
  public function getMessages(): string {
    return is_array($this->messages) ? implode('; ', $this->messages) : (string) $this->messages;
  }

  /**
   * Whether this run was a dry-run.
   */
  public function isDryRun(): bool {
    return $this->dryRun;
  }

}
