<?php

declare(strict_types=1);

namespace Drupal\entity_splitter\Mapping;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface;

/**
 * Builds mapping based on the existing field_group configuration.
 *
 * This is a minimal scaffolding that reads field_group config for a given
 * entity type, bundle and form_mode and returns a basic structure that can be
 * enriched by plugins.
 */
class FieldGroupMapper {

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a field group mapper.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    ModuleHandlerInterface $module_handler,
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    $this->moduleHandler = $module_handler;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Build a mapping skeleton from a field group configuration.
   */
  public function buildMapping(EntitySplitterPluginInterface $entity_splitter_plugin, string $form_mode = 'default'): array {
    $mapping = [
      'field_groups' => [],
      'fields' => [],
      'form_display_components' => [],
    ];

    // Make sure the field group module exists.
    if (!$this->moduleHandler->moduleExists('field_group')) {
      return $mapping;
    }

    // Load the form display for the source entity.
    $form_display = $this->entityTypeManager
      ->getStorage('entity_form_display')
      ->load($entity_splitter_plugin->getSourceEntityTypeId() . '.' . $entity_splitter_plugin->getSourceBundle() . '.' . $form_mode);

    if (!$form_display) {
      return $mapping;
    }

    // Set the form display components.
    $mapping['form_display_components'] = $form_display->getComponents();

    // Get all existing field groups.
    $field_groups = field_group_info_groups($entity_splitter_plugin->getSourceEntityTypeId(), $entity_splitter_plugin->getSourceBundle(), 'form', $form_mode);

    if (empty($field_groups)) {
      return $mapping;
    }

    // Loop through the form field groups.
    foreach ($field_groups as $group_name => $group) {
      // Check if we have ignored field groups.
      $ignored_groups = $entity_splitter_plugin->getIgnoredFieldGroups();

      // We make sure we don't process the ignored groups.
      if (in_array($group_name, $ignored_groups, TRUE)) {
        continue;
      }

      // We set the allowed field groups.
      $mapping['field_groups'][$group_name] = $group;
    }

    return $mapping;
  }

}
