<?php

declare(strict_types=1);

namespace Drupal\entity_splitter\Plugin;

use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\entity_splitter\EntitySplitterResult;

/**
 * Interface for Entity Migration plugins.
 */
interface EntitySplitterPluginInterface extends PluginInspectionInterface {

  /**
   * Get source entity type ID.
   *
   * @return string
   *   Returns the source entity type ID.
   */
  public function getSourceEntityTypeId(): string;

  /**
   * Get a source bundle, if applicable.
   *
   * @return string|null
   *   Returns the source bundle, if applicable.
   */
  public function getSourceBundle(): ?string;

  /**
   * Get target entity type ID.
   *
   * @return string
   *   Returns the target entity type ID.
   */
  public function getTargetEntityTypeId(): string;

  /**
   * Get target bundle entity type ID.
   *
   * @return string
   *   Returns the target bundle entity type ID.
   */
  public function getTargetBundleEntityTypeId(): string;

  /**
   * Get a target bundle, if applicable.
   *
   * @return string|null
   *   Returns the target bundle, if applicable.
   */
  public function getTargetBundle(): ?string;

  /**
   * Get bundle field mapping.
   *
   * @return array
   *   The bundle field mapping.
   */
  public function getBundleFieldMapping(): array;

  /**
   * Get ignored field group ids.
   *
   * @return array|string[]
   *   The (optional) ignored field group ids.
   */
  public function getIgnoredFieldGroups(): array;

  /**
   * Executes the migration structure changes.
   *
   * This function migrates the configuration structure. Things like
   * configuration, form modes ect will be migrated here.
   *
   * The $options may contain a prepared 'mapping' array (from YAML or
   * field_groups) and other flags like 'dry_run'.
   *
   * @param array $options
   *   Arbitrary options to control the entity splitting.
   *
   * @return \Drupal\entity_splitter\EntitySplitterResult
   *   The entity splitter result.
   */
  public function migrateStructure(array $options = []): EntitySplitterResult;

  /**
   * Executes the split data migration.
   *
   * The migrate data function migrates any data defined in the splitter plugin.
   * The $options may contain a prepared 'mapping' array (from YAML or
   * field_groups) and other flags like 'dry_run'.
   *
   * @param array $sandbox
   *   The sandbox.
   * @param array $options
   *   Arbitrary options to control the entity splitting.
   *
   * @return string|\Drupal\Core\StringTranslation\TranslatableMarkup
   *   Returns the migration result.
   */
  public function migrateData(array &$sandbox, array $options = []): string|TranslatableMarkup;

}
