<?php

declare(strict_types=1);

namespace Drupal\entity_splitter\Plugin;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\entity_splitter\EntitySplitterResult;
use Drupal\entity_splitter\Processing\EntitySplitterProcessor;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Entity Migration plugins.
 */
abstract class EntitySplitterBase extends PluginBase implements EntitySplitterPluginInterface, ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * Generic processor that abstracts common migration steps.
   *
   * @var \Drupal\entity_splitter\Processing\EntitySplitterProcessor
   */
  protected EntitySplitterProcessor $processor;

  /**
   * The entity migration processor.
   *
   * @var \Drupal\entity_splitter\Processing\EntitySplitterProcessor
   */
  protected EntitySplitterProcessor $entitySplitterProcessor;

  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Construct an entity migration base.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\entity_splitter\Processing\EntitySplitterProcessor $entity_splitter_processor
   *   The entity migration processor.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager, EntitySplitterProcessor $entity_splitter_processor, LoggerChannelInterface $logger) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->entitySplitterProcessor = $entity_splitter_processor;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('entity_splitter.processor'),
      $container->get('logger.channel.entity_splitter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getSourceEntityTypeId(): string {
    return (string) ($this->pluginDefinition['source_entity_type'] ?? '');
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetEntityTypeId(): string {
    return (string) ($this->pluginDefinition['target_entity_type'] ?? '');
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetBundleEntityTypeId(): string {
    return (string) ($this->pluginDefinition['target_bundle_entity_type'] ?? '');
  }

  /**
   * {@inheritdoc}
   */
  public function getSourceBundle(): ?string {
    $bundle = $this->pluginDefinition['source_bundle'] ?? NULL;
    return $bundle !== NULL ? (string) $bundle : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetBundle(): ?string {
    $bundle = $this->pluginDefinition['target_bundle'] ?? NULL;
    return $bundle !== NULL ? (string) $bundle : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function migrateStructure(array $options = []): EntitySplitterResult {
    // Steps can be controlled via options. By default, execute none.
    $do_create_fields = (bool) ($options['create-fields'] ?? FALSE);
    $do_create_refs = (bool) ($options['create-reference-fields'] ?? FALSE);
    $do_hide_source_fields = (bool) ($options['hide-source-fields'] ?? FALSE);
    $do_remove_source_fields = (bool) ($options['remove-source-fields'] ?? FALSE);
    $do_remove_source_field_groups = (bool) ($options['remove-source-field-groups'] ?? FALSE);
    $do_create_new_target_form_modes = isset($options['create-new-target-form-modes']) && $options['create-new-target-form-modes'] !== '';

    $messages = [];
    $processed = 0;
    $created = 0;
    $updated = 0;
    $failed = 0;

    if (!$options['mapping']) {
      $messages[] = 'No mapping file provided.';
      $failed++;
    }
    else {
      // Execute steps.
      if ($do_create_fields) {
        try {
          $stats = $this->entitySplitterProcessor->createFieldsForBundles($this, $options);
          $messages[] = sprintf('Created %d storages, %d configs across %d bundles.', $stats['storage_count'] ?? 0, $stats['config_count'] ?? 0, $stats['bundle_count'] ?? 0);
          $created += ($stats['config_count'] ?? 0);
        }
        catch (\Throwable $e) {
          $failed++;
          $this->logger->error('Error creating fields: @msg', ['@msg' => $e->getMessage()]);
        }
      }

      // We need to create the reference fields.
      if ($do_create_refs) {
        try {
          $count = $this->entitySplitterProcessor->createReferenceFields($this);
          $messages[] = sprintf('Created %d reference fields.', $count);
          $created += $count;
        }
        catch (\Throwable $e) {
          $failed++;
          $this->logger->error('Error creating reference fields: @msg', ['@msg' => $e->getMessage()]);
        }
      }

      // We need to hide the source fields.
      if ($do_hide_source_fields) {
        try {
          $count = $this->entitySplitterProcessor->hideSourceFields($this, $options);
          $messages[] = sprintf('Hidden %d source fields.', $count);
          $updated += $count;
        }
        catch (\Throwable $e) {
          $failed++;
          $this->logger->error('Error hiding the source fields: @msg', ['@msg' => $e->getMessage()]);
        }
      }

      // We need to remove the source fields.
      if ($do_remove_source_fields) {
        try {
          $count = $this->entitySplitterProcessor->removeSourceFields($this, $options);
          $messages[] = sprintf('Removed %d source fields.', $count);
          $updated += $count;
        }
        catch (\Throwable $e) {
          $failed++;
          $this->logger->error('Error removing the source fields: @msg', ['@msg' => $e->getMessage()]);
        }
      }

      // We need to remove the source fields.
      if ($do_remove_source_field_groups) {
        try {
          $count = $this->entitySplitterProcessor->removeSourceFieldGroups($this, $options);
          $messages[] = sprintf('Removed %d source field groups.', $count);
          $created += $count;
        }
        catch (\Throwable $e) {
          $failed++;
          $this->logger->error('Error removing the source field groups: @msg', ['@msg' => $e->getMessage()]);
        }
      }

      // We need to create new target form modes.
      if ($do_create_new_target_form_modes) {
        try {
          $count = $this->entitySplitterProcessor->createNewTargetFormModes($this, $options['create-new-target-form-modes']);
          $messages[] = sprintf('Created the form mode %s for %d bundles.', $options['create-new-target-form-modes'], $count);
          $created += $count;
        }
        catch (\Throwable $e) {
          $failed++;
          $this->logger->error('Error removing the source field groups: @msg', ['@msg' => $e->getMessage()]);
        }
      }
    }

    // @todo make a dry run option that actually does something.
    return new EntitySplitterResult(
      pluginId: $this->getPluginId(),
      processed: $processed,
      created: $created,
      updated: $updated,
      failed: $failed,
      messages: $messages,
      dryRun: FALSE,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function migrateData(array &$sandbox, array $options = []): string|TranslatableMarkup {
    $field_id = $this->getSourceEntityTypeId() === 'node' ? 'nid' : 'id';
    $entity_storage = $this->entityTypeManager->getStorage($this->getSourceEntityTypeId());

    // Make sure we have the entity storage.
    if (!$entity_storage instanceof EntityStorageInterface) {
      $this->logger->error('Entity storage for @entity_type is not an instance of \Drupal\Core\Entity\EntityStorageInterface', ['@entity_type' => $this->getSourceEntityTypeId()]);
      return '';
    }

    // Initialise sandbox for batch processing.
    if (!isset($sandbox['progress'])) {
      $sandbox['progress'] = 0;
      $sandbox['current_entity'] = 0;
      $sandbox['max'] = $entity_storage->getQuery()
        ->condition('type', $this->getSourceBundle())
        ->accessCheck(FALSE)
        ->count()
        ->execute();
      $sandbox['results'] = [];
    }

    // Process entities in batches if we don't want to process all.
    $batch_size = 10;
    $limit = isset($options['process-all']) && $options['process-all'] ? $sandbox['max'] : $batch_size;

    $entity_ids = $entity_storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('type', $this->getSourceBundle())
      ->condition($field_id, $sandbox['current_entity'], '>')
      ->sort($field_id)
      ->range(0, $limit)
      ->execute();

    // Override with specific entity.
    if (isset($options['entity-id']) && in_array($options['entity-id'], $entity_ids, TRUE)) {
      $sandbox['max'] = 1;
      $entity_ids = [];
      $entity_ids[$options['entity-id']] = $options['entity-id'];
    }

    foreach ($entity_ids as $revision_id => $entity_id) {
      $entity = $entity_storage->load($entity_id);
      if (!$entity instanceof ContentEntityInterface) {
        $sandbox['progress']++;
        $sandbox['current_entity'] = $entity_id;
        continue;
      }
      // Set the current entity id.
      $options['current_entity'] = $entity->id();

      // Migrate the entity either via queue or directly.
      $this->entitySplitterProcessor->migrateFieldData($this, $options);

      $sandbox['progress']++;
      $sandbox['current_entity'] = $entity_id;
    }

    // Update progress.
    if ($sandbox['progress'] < $sandbox['max']) {
      $sandbox['#finished'] = $sandbox['progress'] / $sandbox['max'];
      $message = t('Processed entity of type @type: @current of @max', [
        '@current' => $sandbox['progress'],
        '@max' => $sandbox['max'],
        '@type' => $this->getSourceEntityTypeId(),
      ])->render();
    }
    else {
      $sandbox['#finished'] = 1;

      if ($options['use-queue']) {
        $message = t('Migration prepared added items in the queue for processing')->render();
      }
      else {
        $message = t('Migration completed')->render();
      }
    }

    return $message;
  }

}
