<?php

declare(strict_types=1);

namespace Drupal\entity_splitter\Processing;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\Display\EntityFormDisplayInterface;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\FieldConfigInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueInterface;
use Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\FieldConfigStorage;
use Drupal\field\FieldStorageConfigInterface;
use Drupal\field\FieldStorageConfigStorage;
use Drupal\language\ConfigurableLanguageManagerInterface;
use Drupal\node\NodeInterface;
use Drupal\shs\StringTranslationTrait;

/**
 * Generic processor that performs common entity migration steps.
 *
 * Plugins can delegate to this service so they only need to define mapping
 * and high-level context (source/target entity types, bundle mapping, etc.).
 */
class EntitySplitterProcessor {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The queue.
   *
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected QueueInterface $queue;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Construct a processor.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $entity_field_manager,
    ModuleHandlerInterface $module_handler,
    QueueFactory $queue_factory,
    LanguageManagerInterface $language_manager,
    LoggerChannelInterface $logger,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->moduleHandler = $module_handler;
    $this->queue = $queue_factory->get('entity_splitter_field_data_migration');
    $this->languageManager = $language_manager;
    $this->logger = $logger;
  }

  /**
   * Create fields for target bundles based on field group mapping of a source.
   *
   * Sequence:
   *  - Load source form display and field groups.
   *  - Build label-based group→bundle mapping from target bundles.
   *  - Organize fields per target bundle (preserving group hierarchy where
   *  possible).
   *  - Create/migrate field storages and field configs for target bundles.
   *  - Create target form displays and recreate groups when field_group
   * exists.
   *
   * @param \Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface $entity_splitter_plugin
   *   The entity splitter plugin.
   * @param array $options
   *   The options containing the mapping.
   *
   * @return int[]
   *   Statistics of created items.
   */
  public function createFieldsForBundles(EntitySplitterPluginInterface $entity_splitter_plugin, array $options): array {
    $response = [
      'storage_count' => 0,
      'config_count' => 0,
      'bundle_count' => 0,
      'error_messages' => [],
    ];

    // For now, we only support field_groups logic.
    if (!isset($options['mapping']['field_groups'])) {
      $response['error_messages'][] = 'No field_groups mapping found.';
      return $response;
    }

    // We need the form display components to be set.
    if (!isset($options['mapping']['form_display_components'])) {
      $response['error_messages'][] = 'No form_display_components mapping found.';
    }

    // Set the field group mappings.
    $field_groups = $options['mapping']['field_groups'];
    $form_display_components = $options['mapping']['form_display_components'];

    /** @var \Drupal\field\FieldStorageConfigStorage $field_storage_config_storage */
    $field_storage_config_storage = $this->entityTypeManager->getStorage('field_storage_config');
    /** @var \Drupal\field\FieldConfigStorage $field_config_storage */
    $field_config_storage = $this->entityTypeManager->getStorage('field_config');

    $target_bundles = NULL;
    // If we use bundles, create them if they do not exist yet.
    if ($entity_splitter_plugin->getBundleFieldMapping() && $entity_splitter_plugin->getTargetBundleEntityTypeId() !== '') {
      $target_bundles = $this->createBundleTypes($entity_splitter_plugin);
    }

    // Source and target definitions.
    $source_entity_type = $entity_splitter_plugin->getSourceEntityTypeId();
    $source_bundle = $entity_splitter_plugin->getSourceBundle();
    $target_entity_type = $entity_splitter_plugin->getTargetEntityTypeId();

    $bundle_group_mapping = $this->createLabelBasedGroupMapping($field_groups, $target_bundles);

    // Get existing fields.
    $existing_fields = $this->entityFieldManager->getFieldDefinitions($source_entity_type, $source_bundle);

    // Organise data per bundle, including group hierarchy.
    $bundle_data = $this->organizeFieldsAndGroupsByLabel($field_groups, $bundle_group_mapping, $existing_fields, $form_display_components);

    // Migrate field storages (once per field).
    $migrated_field_storages = $this->migrateFieldStorage($bundle_data, $field_storage_config_storage, $source_entity_type, $target_entity_type);

    // Create field configs for each bundle.
    $total_field_configs = $this->createFieldConfigsForBundles($bundle_data, $field_config_storage, $source_entity_type, $source_bundle, $target_entity_type);

    // Create form displays including field groups.
    $this->createBundleDisplaysWithGroups($bundle_data, $target_entity_type);

    // Set the statistics.
    $response['storage_count'] = count($migrated_field_storages);
    $response['config_count'] = $total_field_configs;
    $response['bundle_count'] = count($bundle_data);

    return $response;
  }

  /**
   * Creates reference fields on the source bundle to link to target bundles.
   *
   * @param \Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface $entity_splitter_plugin
   *   The entity splitter plugin.
   *
   * @return int
   *   Returns the number of reference fields created.
   */
  public function createReferenceFields(EntitySplitterPluginInterface $entity_splitter_plugin): int {
    $created_fields = 0;

    /** @var \Drupal\field\FieldStorageConfigStorage $field_storage_config_storage */
    $field_storage_config_storage = $this->entityTypeManager->getStorage('field_storage_config');
    /** @var \Drupal\field\FieldConfigStorage $field_config_storage */
    $field_config_storage = $this->entityTypeManager->getStorage('field_config');

    foreach ($entity_splitter_plugin->getBundleFieldMapping() as $bundle_id => $bundle_info) {
      if (!isset($bundle_info['reference_field'])) {
        continue;
      }

      $field_name = $bundle_info['reference_field'];

      // Create field storage.
      if (!$field_storage_config_storage->load($entity_splitter_plugin->getSourceEntityTypeId() . '.' . $field_name)) {
        try {
          $field_storage = $field_storage_config_storage->create([
            'field_name' => $field_name,
            'entity_type' => $entity_splitter_plugin->getSourceEntityTypeId(),
            'type' => 'entity_reference',
            'cardinality' => 1,
            'settings' => [
              'target_type' => $entity_splitter_plugin->getTargetEntityTypeId(),
            ],
          ]);
          $field_storage->save();
        }
        catch (\Exception $e) {
          $this->logger->error('Error creating field storage @field: @error', [
            '@field' => $field_name,
            '@error' => $e->getMessage(),
          ]);
          continue;
        }
      }

      // Create field config.
      if (!$field_config_storage->load($entity_splitter_plugin->getSourceEntityTypeId() . '.' . $entity_splitter_plugin->getSourceBundle() . '.' . $field_name)) {
        try {
          $field_config = $field_config_storage->create([
            'field_name' => $field_name,
            'entity_type' => $entity_splitter_plugin->getSourceEntityTypeId(),
            'bundle' => $entity_splitter_plugin->getSourceBundle(),
            'label' => $this->convertMachineNameToLabel($bundle_id),
            'description' => 'Reference to ' . $this->convertMachineNameToLabel($bundle_id) . ' data',
            'required' => FALSE,
            'settings' => [
              'handler' => 'default:' . $entity_splitter_plugin->getTargetEntityTypeId(),
              'handler_settings' => [
                'target_bundles' => [$bundle_id => $bundle_id],
                'sort' => [
                  'field' => '_none',
                ],
                'auto_create' => FALSE,
                'auto_create_bundle' => '',
              ],
            ],
          ]);
          $field_config->save();

          $created_fields++;
        }
        catch (\Exception $e) {
          $this->logger->error('Error creating field config @field: @error', [
            '@field' => $field_name,
            '@error' => $e->getMessage(),
          ]);
        }
      }
    }

    return $created_fields;
  }

  /**
   * Create new target form modes.
   *
   * @param \Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface $entity_splitter_plugin
   *   The entity splitter plugin.
   * @param string $form_mode_name
   *   The form mode name.
   *
   * @return int
   *   Returns the number of created form modes.
   */
  public function createNewTargetFormModes(EntitySplitterPluginInterface $entity_splitter_plugin, string $form_mode_name): int {
    $created = 0;

    if (!$this->moduleHandler->moduleExists('inline_entity_form')) {
      return $created;
    }

    $messages = [];

    $source_entity_type = $entity_splitter_plugin->getSourceEntityTypeId();
    $source_bundle = $entity_splitter_plugin->getSourceBundle();

    // Load the default form display.
    $default_form_display = $this->entityTypeManager
      ->getStorage('entity_form_display')
      ->load($source_entity_type . '.' . $source_bundle . '.default');

    if (!$default_form_display) {
      return $created;
    }

    // Hide all reference fields from the default form display.
    foreach ($entity_splitter_plugin->getBundleFieldMapping() as $bundle_id => $bundle_info) {
      if (!isset($bundle_info['reference_field'])) {
        continue;
      }
      $reference_field_name = $bundle_info['reference_field'];
      if ($default_form_display->getComponent($reference_field_name)) {
        $default_form_display->removeComponent($reference_field_name);
        $messages[] = $this->t('Hidden @bundle : @field from default form display', [
          '@bundle' => $bundle_id,
          '@field' => $reference_field_name,
        ]);
      }
    }
    $default_form_display->save();

    // First create the custom target form modes.
    $custom_target_form_modes_result = $this->createCustomTargetFormMode($entity_splitter_plugin, $form_mode_name);
    if ($custom_target_form_modes_result) {
      $messages[] = $custom_target_form_modes_result;
    }

    // Create form modes for each bundle.
    foreach ($entity_splitter_plugin->getBundleFieldMapping() as $bundle_id => $bundle_info) {
      if (!isset($bundle_info['reference_field'])) {
        continue;
      }
      $reference_field_name = $bundle_info['reference_field'];
      $form_mode_id = $this->getEntitySplitterFormModeFromBundle($bundle_id, $entity_splitter_plugin->getSourceBundle());

      // Create the form mode entity if it doesn't exist.
      $form_mode = $this->entityTypeManager
        ->getStorage('entity_form_mode')
        ->load($source_entity_type . '.' . $form_mode_id);

      if (!$form_mode) {
        $form_mode = $this->entityTypeManager
          ->getStorage('entity_form_mode')
          ->create([
            'id' => $source_entity_type . '.' . $form_mode_id,
            'label' => $this->convertLabelByMachineName($bundle_id),
            'targetEntityType' => $source_entity_type,
            'status' => TRUE,
          ]);
        $form_mode->save();
        $created++;
        $messages[] = $this->t('Created form mode: @mode', ['@mode' => $form_mode_id]);
      }

      // Create the form display for this mode.
      $form_display_id = $source_entity_type . '.' . $source_bundle . '.' . $form_mode_id;

      $form_display = $this->entityTypeManager
        ->getStorage('entity_form_display')
        ->load($form_display_id);

      if (!$form_display instanceof EntityFormDisplay) {
        // Create a new form display based on the default one.
        /** @var \Drupal\Core\Entity\Display\EntityFormDisplayInterface $form_display */
        $form_display = $default_form_display->createDuplicate();
        $form_display->set('id', $form_display_id);
        $form_display->set('mode', $form_mode_id);
        $form_display->set('status', TRUE);
      }

      // Hide all components except the specific reference field.
      $components = $form_display->getComponents();
      foreach ($components as $component_name => $component_config) {
        if ($component_name !== $reference_field_name) {
          $form_display->removeComponent($component_name);
        }
      }

      // Configure the reference field as an inline entity form.
      $form_display->setComponent($reference_field_name, [
        'type' => 'inline_entity_form_simple',
        'weight' => 0,
        'settings' => [
          'form_mode' => $form_mode_name,
          'revision' => TRUE,
        ],
        'third_party_settings' => [],
      ]);

      // Remove any field groups for this form display.
      $this->removeFieldGroupsFromFormDisplay($form_display);

      $form_display->save();
      $messages[] = $this->t('Created/updated form display for @bundle with inline entity form', ['@bundle' => $bundle_id]);
    }

    // Set the messages.
    $this->logger->info('Created/updated form modes for @source_entity_type: @messages', [
      '@source_entity_type' => $source_entity_type,
      '@messages' => implode(', ', $messages),
    ]);

    return $created;
  }

  /**
   * Migrate field data from the source bundle to target entities.
   *
   * @param \Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface $entity_splitter_plugin
   *   The entity splitter plugin.
   * @param array $options
   *   The options array.
   */
  public function migrateFieldData(EntitySplitterPluginInterface $entity_splitter_plugin, array $options): void {
    $do_queue_items = (bool) ($options['use-queue'] ?? FALSE);
    $remove_existing_references = (bool) ($options['remove-existing-references'] ?? FALSE);

    // Make sure we have a current entity id to process.
    if (!isset($options['current_entity'])) {
      return;
    }

    $entity_id = (int) $options['current_entity'];

    if ($do_queue_items) {
      $this->queue->createItem([
        'entity_id' => $entity_id,
        'plugin_id' => $entity_splitter_plugin->getPluginId(),
        'remove-existing-references' => $remove_existing_references,
      ]);
    }
    else {
      $this->migrateEntity($entity_splitter_plugin, $entity_id, $remove_existing_references);
    }
  }

  /**
   * Migrate the entity to the target entity.
   *
   * @param \Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface $entity_splitter_plugin
   *   The entity splitter plugin.
   * @param int $entity_id
   *   The entity id.
   * @param bool $remove_existing_references
   *   Remove existing references.
   */
  public function migrateEntity(EntitySplitterPluginInterface $entity_splitter_plugin, int $entity_id, bool $remove_existing_references = FALSE): void {
    $migrations_for_entity = 0;
    $deletions_for_entity = 0;
    $entity = $this->entityTypeManager->getStorage($entity_splitter_plugin->getSourceEntityTypeId())->load($entity_id);

    if (!$entity instanceof ContentEntityInterface) {
      return;
    }

    // Get the target entity storage.
    $target_data_storage = $this->entityTypeManager->getStorage($entity_splitter_plugin->getTargetEntityTypeId());

    // Get all available translations for this node.
    $available_languages = $entity->getTranslationLanguages();

    // Remove any existing references.
    if ($remove_existing_references) {
      // Process each bundle mapping.
      $referenced_entities_to_delete = [];
      foreach ($entity_splitter_plugin->getBundleFieldMapping() as $bundle_id => $bundle_info) {
        if (!isset($bundle_info['reference_field'])) {
          continue;
        }

        // Set the reference field name.
        $reference_field_name = $bundle_info['reference_field'];

        foreach ($available_languages as $langcode => $language) {
          $translated_entity = $entity->getTranslation($langcode);

          // We save the empty value and delete the existing reference.
          if ($translated_entity->hasField($reference_field_name) && !$translated_entity->get($reference_field_name)->isEmpty()) {
            $referenced_entities_to_delete[] = $translated_entity->get($reference_field_name)->entity;
            $translated_entity->set($reference_field_name, NULL);
            $deletions_for_entity++;
          }
        }
      }

      if ($deletions_for_entity > 0) {
        // Update all translations of the node with removed reference fields.
        foreach ($available_languages as $langcode => $language) {
          $translated_entity = $entity->getTranslation($langcode);
          $translated_entity->save();
        }

        // Remove the actual referenced entities.
        foreach ($referenced_entities_to_delete as $referenced_entity) {
          if (!$referenced_entity instanceof ContentEntityInterface) {
            continue;
          }
          $referenced_entity->delete();
        }
      }
    }

    // Process each bundle mapping.
    foreach ($entity_splitter_plugin->getBundleFieldMapping() as $bundle_id => $bundle_info) {
      if (!isset($bundle_info['reference_field'])) {
        continue;
      }

      // Set the reference field name.
      $reference_field_name = $bundle_info['reference_field'];

      // Process each translation - create a separate entity for each
      // language.
      foreach ($available_languages as $langcode => $language) {
        $translated_entity = $entity->getTranslation($langcode);

        // Get the field data for this subentity in this language.
        $field_data = $this->extractFieldDataForBundle($entity_splitter_plugin, $translated_entity, $bundle_id);

        // Skip if no data found for this bundle in this language.
        if (empty($field_data)) {
          continue;
        }

        // Get the title of the entity.
        $title = $entity instanceof NodeInterface ? $entity->getTitle() : $entity->label();

        // Create a separate entity for each language.
        /** @var \Drupal\Core\Entity\ContentEntityInterface $target_entity */
        $target_entity = $target_data_storage->create([
          'title' => $title . ' - ' . $this->convertMachineNameToLabel($bundle_id),
          'bundle' => $bundle_id,
          'langcode' => $langcode,
        ]);

        // Populate the target entity with field data.
        foreach ($field_data as $field_name => $field_value) {
          if ($target_entity->hasField($field_name)) {
            $target_entity->set($field_name, $field_value);
          }
        }

        // Save the target entity.
        try {
          $target_entity->save();

          // Update the reference field on the translated entities.
          if ($translated_entity->hasField($reference_field_name)) {
            $translated_entity->set($reference_field_name, $target_entity->id());
          }

          $migrations_for_entity++;
        }
        catch (\Exception $e) {
          $this->logger->error('Error creating @target_entity_type for bundle @bundle, @type @id, language @langcode: @error', [
            '@target_entity_type' => $entity_splitter_plugin->getTargetEntityTypeId(),
            '@bundle' => $bundle_id,
            '@type' => $entity->getEntityTypeId(),
            '@id' => $entity->id(),
            '@langcode' => $langcode,
            '@error' => $e->getMessage(),
          ]);
        }
      }
    }

    // Save all translations of the node with updated reference fields.
    if ($migrations_for_entity > 0) {
      foreach ($available_languages as $langcode => $language) {
        $translated_entity = $entity->getTranslation($langcode);
        $translated_entity->save();
      }
    }
  }

  /**
   * Hide the source fields.
   *
   * @param \Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface $entity_splitter_plugin
   *   The entity splitter plugin.
   * @param array $options
   *   The options.
   *
   * @return int
   *   Returns the number of hidden source fields.
   */
  public function hideSourceFields(EntitySplitterPluginInterface $entity_splitter_plugin, array $options = []): int {
    // Collect all fields to hide based on your bundle mapping.
    $fields_to_hide = [];

    // Get the field groups.
    $reference_fields = [];
    $field_groups = $options['mapping']['field_groups'];
    foreach ($entity_splitter_plugin->getBundleFieldMapping() as $bundle_id => $bundle_info) {
      $bundle_fields = $this->getBundleFieldNames($field_groups, $bundle_id);
      array_push($fields_to_hide, ...$bundle_fields);
      $reference_fields[] = $bundle_info['reference_field'];
    }

    // Remove duplicates and exclude the new reference fields.
    $fields_to_hide = array_unique($fields_to_hide);
    $fields_to_hide = array_diff($fields_to_hide, array_values($reference_fields));

    $hidden_fields = [];

    // Load the form display.
    $form_display = $this->entityTypeManager->getStorage('entity_form_display')->load($entity_splitter_plugin->getSourceEntityTypeId() . '.' . $entity_splitter_plugin->getSourceBundle() . '.default');
    if ($form_display) {
      // Process all fields at once.
      foreach ($fields_to_hide as $field_name) {
        try {
          // Remove the field component from the form display (hides it).
          $form_display->removeComponent($field_name);
          $hidden_fields[] = $field_name;
        }
        catch (\Exception $e) {
          $this->logger->error('Error hiding field @field: @error', [
            '@field' => $field_name,
            '@error' => $e->getMessage(),
          ]);
        }
      }

      // Save the form display after all fields have been processed.
      $form_display->save();
    }

    // Clear caches after form display update.
    $this->entityFieldManager->clearCachedFieldDefinitions();

    return count($hidden_fields);
  }

  /**
   * Remove the source fields.
   *
   * @param \Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface $entity_splitter_plugin
   *   The entity splitter plugin.
   * @param array $options
   *   The configuration options.
   *
   * @return int
   *   Returns the number of removed source fields.
   */
  public function removeSourceFields(EntitySplitterPluginInterface $entity_splitter_plugin, array $options = []): int {
    $field_config_storage = $this->entityTypeManager->getStorage('field_config');
    $field_storage_config_storage = $this->entityTypeManager->getStorage('field_storage_config');

    // Get the field groups.
    $field_groups = $options['mapping']['field_groups'];

    // Collect all fields to delete based on your bundle mapping.
    $reference_fields = [];
    $fields_to_delete = [];
    foreach ($entity_splitter_plugin->getBundleFieldMapping() as $bundle_id => $bundle_info) {
      $bundle_fields = $this->getBundleFieldNames($field_groups, $bundle_id);
      array_push($fields_to_delete, ...$bundle_fields);
      $reference_fields[] = $bundle_info['reference_field'];
    }

    // Remove duplicates and exclude the new reference fields.
    $fields_to_delete = array_unique($fields_to_delete);
    $fields_to_delete = array_diff($fields_to_delete, array_values($reference_fields));

    $deleted_fields = [];

    // Process all fields at once.
    foreach ($fields_to_delete as $field_name) {
      try {
        // Delete field config first.
        $field_config = $field_config_storage->load("{$entity_splitter_plugin->getSourceEntityTypeId()}.{$entity_splitter_plugin->getSourceBundle()}.{$field_name}");
        if ($field_config) {
          $field_config->delete();
          $deleted_fields[] = $field_name;
        }

        // Delete field storage if no other bundles use it.
        $field_storage = $field_storage_config_storage->load("{$entity_splitter_plugin->getSourceEntityTypeId()}.{$field_name}");
        if ($field_storage) {
          // Check if other bundles use this field.
          $field_configs = $field_config_storage->loadByProperties(['field_name' => $field_name]);
          if (empty($field_configs)) {
            $field_storage->delete();
          }
        }
      }
      catch (\Exception $e) {
        $this->logger->error('Error deleting field @field: @error', [
          '@field' => $field_name,
          '@error' => $e->getMessage(),
        ]);
      }
    }

    // Clear caches after field deletion.
    $this->entityFieldManager->clearCachedFieldDefinitions();

    return count($deleted_fields);
  }

  /**
   * Remove the source field groups.
   *
   * @param \Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface $entity_splitter_plugin
   *   The entity splitter plugin.
   * @param array $options
   *   The options array.
   *
   * @return int
   *   Returns the number of removed source field groups.
   */
  public function removeSourceFieldGroups(EntitySplitterPluginInterface $entity_splitter_plugin, array $options = []): int {
    // Source and target definitions.
    $source_entity_type = $entity_splitter_plugin->getSourceEntityTypeId();
    $source_bundle = $entity_splitter_plugin->getSourceBundle();

    $form_mode = $options['form-mode'] ?? 'default';

    // Load the form display for the source entity.
    $form_display = $this->entityTypeManager
      ->getStorage('entity_form_display')
      ->load($source_entity_type . '.' . $source_bundle . '.' . $form_mode);

    if (!$form_display) {
      return 0;
    }

    // Get all existing field groups.
    $form_field_groups = [];
    if ($this->moduleHandler->moduleExists('field_group')) {
      $form_field_groups = field_group_info_groups($source_entity_type, $source_bundle, 'form', $form_mode);
    }

    // Get all existing field definitions for this bundle.
    $field_definitions = $this->entityFieldManager->getFieldDefinitions($source_entity_type, $source_bundle);
    $existing_field_names = array_keys($field_definitions);

    // Process field groups that contain non-existing fields.
    $deleted_groups = [];
    $cleaned_groups = [];

    foreach ($form_field_groups as $group_name => $group_info) {
      if (empty($group_info->children)) {
        continue;
      }

      $valid_children = [];
      $invalid_children = [];

      // Separate valid and invalid fields.
      foreach ($group_info->children as $field_name) {
        if (in_array($field_name, $existing_field_names) && !is_null($form_display->getComponent($field_name))) {
          $valid_children[] = $field_name;
        }
        else {
          $invalid_children[] = $field_name;
        }
      }

      // Only proceed if there are invalid children.
      if (!empty($invalid_children)) {
        if (empty($valid_children)) {
          // Group has no valid fields left - delete it entirely.
          $form_display->removeComponent($group_name);
          $form_display->unsetThirdPartySetting('field_group', $group_name);
          $deleted_groups[] = $group_name;
        }
        else {
          // Group has some valid fields - update to keep only valid ones.
          $third_party_settings = $form_display->getThirdPartySettings('field_group');
          if (isset($third_party_settings[$group_name])) {
            $third_party_settings[$group_name]['children'] = $valid_children;
            $form_display->setThirdPartySetting('field_group', $group_name, $third_party_settings[$group_name]);
            $cleaned_groups[] = $group_name . ' (removed: ' . implode(', ', $invalid_children) . ')';
          }
        }
      }
    }

    // Save the form display if changes were made.
    if (!empty($deleted_groups) || !empty($cleaned_groups)) {
      $form_display->save();
    }

    return count($cleaned_groups);
  }

  /**
   * Create bundle types.
   *
   * @param \Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface $entity_splitter_plugin
   *   The entity splitter plugin.
   *
   * @return array
   *   Returns the array of bundle IDs.
   */
  private function createBundleTypes(EntitySplitterPluginInterface $entity_splitter_plugin): array {
    $target_bundle_storage = $this->entityTypeManager->getStorage($entity_splitter_plugin->getTargetBundleEntityTypeId());

    // Initialize the array of bundle IDs.
    $entity_bundles = [];

    foreach ($entity_splitter_plugin->getBundleFieldMapping() as $bundle_id => $bundle_info) {
      // Check if the bundle already exists.
      $existing_bundle = $target_bundle_storage->load($bundle_id);
      if (!$existing_bundle) {
        // Create the bundle type in English.
        $bundle = $target_bundle_storage->create([
          'id' => $bundle_id,
          'label' => $bundle_info['label'],
          'description' => $bundle_info['description'],
          'language' => 'en',
        ]);

        $bundle->save();
        $entity_bundles[] = $bundle_id;

        // Clear the cache after creating each bundle.
        $this->entityFieldManager->clearCachedFieldDefinitions();
      }
      else {
        $entity_bundles[] = $existing_bundle->id();
      }
    }

    return $entity_bundles;
  }

  /**
   * Creates a label-based mapping structure for field groups to bundles.
   *
   * Uses group labels converted to machine names to determine the target
   * bundle.
   *
   * @param array $field_groups
   *   The field groups from the source entity.
   * @param array $target_bundles
   *   Available target bundle machine names.
   *
   * @return array
   *   Mapping structure: [bundle_name][group_name] = group_info.
   */
  private function createLabelBasedGroupMapping(array $field_groups, array $target_bundles): array {
    $bundle_group_mapping = [];

    foreach ($field_groups as $group_name => $group_info) {
      // Convert label to machine name and check if it matches a target bundle.
      $converted_bundle_name = $this->convertLabelByMachineName($group_info->label);

      if (in_array($converted_bundle_name, $target_bundles)) {
        $target_bundle = $converted_bundle_name;
      }
      else {
        // Try to find a bundle by checking parent hierarchy.
        $target_bundle = $this->findBundleByLabelHierarchy($group_name, $field_groups, $target_bundles);
      }

      if ($target_bundle) {
        // Initialise bundle mapping if not exists.
        if (!isset($bundle_group_mapping[$target_bundle])) {
          $bundle_group_mapping[$target_bundle] = [];
        }

        $bundle_group_mapping[$target_bundle][$group_name] = $group_info;
      }
      else {
        $this->logger->warning('Could not determine bundle for group @group (label: "@label" -> "@converted")', [
          '@group' => $group_name,
          '@label' => $group_info->label,
          '@converted' => $converted_bundle_name,
        ]);
      }
    }

    return $bundle_group_mapping;
  }

  /**
   * Create a custom form mode for a target entity type.
   *
   * @param \Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface $entity_splitter_plugin
   *   The entity splitter plugin.
   * @param string $form_mode_id
   *   The form mode.
   *
   * @return string
   *   Returns the results of the form mode creation.
   */
  private function createCustomTargetFormMode(EntitySplitterPluginInterface $entity_splitter_plugin, string $form_mode_id): string {
    $messages = [];

    // Create the form mode for the target entity type.
    $entity_type = $entity_splitter_plugin->getTargetEntityTypeId();

    $form_mode = $this->entityTypeManager
      ->getStorage('entity_form_mode')
      ->load($entity_type . '.' . $form_mode_id);

    if (!$form_mode) {
      $form_mode = $this->entityTypeManager
        ->getStorage('entity_form_mode')
        ->create([
          'id' => $entity_type . '.' . $form_mode_id,
          'label' => $this->convertMachineNameToLabel($form_mode_id),
          'targetEntityType' => $entity_type,
          'status' => TRUE,
          'cache' => TRUE,
          'dependencies' => [
            'module' => ['entity_splitter'],
          ],
        ]);
      $form_mode->save();
      $messages[] = $this->t('Created form mode: @entity_type.@mode', [
        '@entity_type' => $entity_type,
        '@mode' => $form_mode_id,
      ]);
    }

    // Create a form display for each bundle.
    foreach ($entity_splitter_plugin->getBundleFieldMapping() as $bundle_id => $bundle_info) {
      $form_display_id = $entity_type . '.' . $bundle_id . '.' . $form_mode_id;

      $form_display = $this->entityTypeManager
        ->getStorage('entity_form_display')
        ->load($form_display_id);

      if (!$form_display) {
        // Load the default form display for this bundle.
        $default_form_display = $this->entityTypeManager
          ->getStorage('entity_form_display')
          ->load($entity_type . '.' . $bundle_id . '.default');

        if ($default_form_display) {
          // Create a new form display based on the default.
          $form_display = $default_form_display->createDuplicate();
          $form_display->set('id', $form_display_id);
          $form_display->set('mode', $form_mode_id);
          $form_display->set('status', TRUE);

          // We hide the title translation and langcode for this form mode.
          $hide_fields = [
            'title',
            'translation',
            'langcode',
          ];

          // Hide the title and the language fields.
          $components = $form_display->getComponents();
          foreach ($components as $component_name => $component_config) {
            if (in_array($component_name, $hide_fields, TRUE)) {
              $form_display->removeComponent($component_name);
            }
          }

          $form_display->save();

          $messages[] = $this->t('Created form display: @display_id', [
            '@display_id' => $form_display_id,
          ]);
        }
        else {
          $messages[] = $this->t('Warning: Default form display not found for @entity_type.@bundle', [
            '@entity_type' => $entity_type,
            '@bundle' => $bundle_id,
          ]);
        }
      }
    }

    return implode(' | ', $messages);
  }

  /**
   * Creates field configs for all bundles.
   *
   * @param array $bundle_data
   *   Organized bundle data.
   * @param \Drupal\field\FieldConfigStorage $field_config_storage
   *   Field config storage.
   * @param string $source_entity_type
   *   Source entity type.
   * @param string $source_bundle
   *   Source bundle.
   * @param string $target_entity_type
   *   Target entity type.
   *
   * @return int
   *   Number of field configs created.
   */
  private function createFieldConfigsForBundles(array $bundle_data, FieldConfigStorage $field_config_storage, string $source_entity_type, string $source_bundle, string $target_entity_type): int {
    $total_configs = 0;

    foreach ($bundle_data as $bundle => $data) {
      foreach ($data['fields'] as $field_name => $field_info) {
        // Skip if field config already exists for this bundle.
        if ($field_config_storage->load($target_entity_type . '.' . $bundle . '.' . $field_name)) {
          continue;
        }

        $original_field_config = $field_config_storage->load($source_entity_type . '.' . $source_bundle . '.' . $field_name);
        if (!$original_field_config) {
          continue;
        }

        try {
          $new_field_config = $field_config_storage->create([
            'field_name' => $field_name,
            'entity_type' => $target_entity_type,
            'bundle' => $bundle,
            'label' => $original_field_config->getLabel(),
            'description' => $original_field_config->getDescription(),
            'required' => $original_field_config->isRequired(),
            'default_value' => $original_field_config->getDefaultValueLiteral(),
            'settings' => $original_field_config->getSettings(),
            // Keep the base translation aligned with the source config.
            'translatable' => $original_field_config->isTranslatable(),
            'langcode' => $original_field_config->language()->getId(),
          ]);
          $new_field_config->save();

          // Copy translations for the FieldConfig label/description and
          // settings if any.
          $this->migrateFieldConfigTranslations($original_field_config, $new_field_config, 'field.field.');

          $total_configs++;
        }
        catch (\Exception $e) {
          $this->logger->error('Error creating field config @field for bundle @bundle: @error', [
            '@field' => $field_name,
            '@bundle' => $bundle,
            '@error' => $e->getMessage(),
          ]);
        }
      }
    }

    return $total_configs;
  }

  /**
   * Migrates translations for FieldConfig or fieldStorageConfig.
   *
   * Copies translated label and description or settings from the source
   * FieldConfig to the target FieldConfig for all non-default languages that
   * exist on the source but not yet on the target.
   *
   * @param \Drupal\field\FieldStorageConfigInterface|\Drupal\Core\Field\FieldConfigInterface $source_config
   *   The source field config.
   * @param \Drupal\field\FieldStorageConfigInterface|\Drupal\Core\Field\FieldConfigInterface $target_config
   *   The target field config.
   * @param string $config_prefix
   *   The config prefix.
   */
  private function migrateFieldConfigTranslations(FieldStorageConfigInterface|FieldConfigInterface $source_config, FieldStorageConfigInterface|FieldConfigInterface $target_config, string $config_prefix): void {
    $languages = $this->languageManager->getLanguages();

    foreach ($languages as $langcode => $language) {
      // Skip the original target field config language.
      if (
        $source_config->language()->getId() === $langcode ||
        !$this->languageManager instanceof ConfigurableLanguageManagerInterface
      ) {
        continue;
      }

      // Get the source translation override.
      $source_override = $this->languageManager->getLanguageConfigOverride($langcode, $config_prefix . $source_config->getConfigTarget());

      // Skip if no translation exists for this language.
      if (empty($source_override->get())) {
        continue;
      }

      // Get the target translation override.
      $target_override = $this->languageManager->getLanguageConfigOverride($langcode, $config_prefix . $target_config->getConfigTarget());

      // Get all translatable data from source.
      $source_data = $source_override->get();

      // Only copy specific translatable properties to avoid copying
      // non-translatable settings.
      $translatable_properties = ['label', 'description', 'settings'];

      foreach ($translatable_properties as $property) {
        if (isset($source_data[$property])) {
          $target_override->set($property, $source_data[$property]);
        }
      }

      // Save the target override.
      $target_override->save();
    }
  }

  /**
   * Creates form displays including field groups for all bundles.
   *
   * @param array $bundle_data
   *   Organized bundle data.
   * @param string $target_entity_type
   *   Target entity type.
   */
  private function createBundleDisplaysWithGroups(array $bundle_data, string $target_entity_type): void {
    $form_display_storage = $this->entityTypeManager->getStorage('entity_form_display');

    foreach ($bundle_data as $bundle => $data) {
      $this->logger->info('Processing bundle @bundle with @group_count groups and @field_count fields', [
        '@bundle' => $bundle,
        '@group_count' => count($data['groups']),
        '@field_count' => count($data['fields']),
      ]);

      // Create or load form display.
      $form_display_id = $target_entity_type . '.' . $bundle . '.default';
      $form_display = $form_display_storage->load($form_display_id);

      if (!$form_display) {
        $form_display = $form_display_storage->create([
          'targetEntityType' => $target_entity_type,
          'bundle' => $bundle,
          'mode' => 'default',
        ]);
      }

      // Add field components.
      $weight = 0;
      foreach ($data['fields'] as $field_name => $field_info) {
        $component = $field_info['form_component'];
        $component['weight'] = $weight++;
        $form_display->setComponent($field_name, $component);
      }

      $form_display->save();

      // Recreate field groups with hierarchy.
      if ($this->moduleHandler->moduleExists('field_group')) {
        $this->recreateFieldGroupsByBundle($target_entity_type, $bundle, $data['groups'], $data['fields'], $data['group_hierarchy']);
      }

      $this->logger->info('Created display for bundle @bundle with @field_count fields and @group_count groups', [
        '@bundle' => $bundle,
        '@field_count' => count($data['fields']),
        '@group_count' => count($data['groups']),
      ]);
    }
  }

  /**
   * Converts a label to a machine name format.
   *
   * @param string $label
   *   The human-readable label.
   *
   * @return string
   *   Returns the machine name.
   */
  private function convertLabelByMachineName(string $label): string {
    // Convert to lowercase.
    $machine_name = strtolower($label);

    // Replace spaces and special characters with underscores.
    $machine_name = preg_replace('/[^a-z0-9]+/', '_', $machine_name);

    // Remove leading/trailing underscores.
    $machine_name = trim($machine_name, '_');

    // Replace multiple consecutive underscores with a single underscore.
    return preg_replace('/_+/', '_', $machine_name);
  }

  /**
   * Converts bundle machine name to human-readable label.
   *
   * @param string $bundle
   *   Bundle machine name.
   *
   * @return string
   *   Human-readable label.
   */
  private function convertMachineNameToLabel(string $bundle): string {
    // Convert underscores to spaces and capitalize.
    $label = str_replace('_', ' ', $bundle);
    return ucwords($label);
  }

  /**
   * Get entity splitter form mode from bundle name.
   *
   * @param string $bundle
   *   The bundle name.
   * @param string $source_bundle
   *   The source bundle.
   *
   * @return string
   *   Returns the form mode from the bundle.
   */
  private function getEntitySplitterFormModeFromBundle(string $bundle, string $source_bundle): string {
    return $bundle . '_' . $source_bundle;
  }

  /**
   * Get field names that belong to a bundle based on field group mapping.
   *
   * @param array $field_groups
   *   The field groups.
   * @param string $target_bundle
   *   The target bundle to get field names for.
   *
   * @return array
   *   Array of field names belonging to this bundle.
   */
  private function getBundleFieldNames(array $field_groups, string $target_bundle): array {
    $field_names = [];

    if (!$this->moduleHandler->moduleExists('field_group')) {
      return $field_names;
    }

    foreach ($field_groups as $group_name => $group_info) {
      // Check if this group maps to our target bundle.
      $converted_bundle_name = $this->convertLabelByMachineName($group_info->label);

      if ($converted_bundle_name === $target_bundle) {
        $group_bundle = $target_bundle;
      }
      else {
        // Check the parent hierarchy for bundle mapping.
        $group_bundle = $this->findBundleByLabelHierarchy($group_name, $field_groups, [$target_bundle]);
      }

      // If this group maps to our target bundle, collect its field names.
      if ($group_bundle === $target_bundle) {
        foreach ($group_info->children as $field_name) {
          // Skip if the field is actually another group.
          if (isset($field_groups[$field_name])) {
            continue;
          }

          $field_names[] = $field_name;
        }
      }
    }

    return array_unique($field_names);
  }

  /**
   * Organises fields and groups per bundle using label-based mapping.
   *
   * @param array $field_groups
   *   The field groups from the source entity.
   * @param array $bundle_group_mapping
   *   Bundle group mapping.
   * @param array $existing_fields
   *   Existing field definitions.
   * @param array $form_components
   *   Form display components.
   *
   * @return array
   *   Organised data structure: [bundle]['fields'|'groups'|'group_hierarchy'].
   */
  private function organizeFieldsAndGroupsByLabel(array $field_groups, array $bundle_group_mapping, array $existing_fields, array $form_components): array {
    $bundle_data = [];

    foreach ($bundle_group_mapping as $bundle => $groups) {
      // Initialize bundle data.
      if (!isset($bundle_data[$bundle])) {
        $bundle_data[$bundle] = [
          'fields' => [],
          'groups' => [],
          'group_hierarchy' => [],
        ];
      }

      foreach ($groups as $group_name => $group_info) {

        // Store group information.
        $bundle_data[$bundle]['groups'][$group_name] = $group_info;
        $bundle_data[$bundle]['group_hierarchy'][$group_name] = $this->buildGroupHierarchyPathByLabels($group_name, $field_groups);

        // Add fields from this group.
        foreach ($group_info->children as $child_name) {

          // Check if a child is a field.
          if (isset($existing_fields[$child_name]) &&
            !$existing_fields[$child_name]->getFieldStorageDefinition()->isBaseField() &&
            isset($form_components[$child_name])) {

            $bundle_data[$bundle]['fields'][$child_name] = [
              'definition' => $existing_fields[$child_name],
              'form_component' => $form_components[$child_name],
              'group' => $group_name,
              'group_hierarchy' => $bundle_data[$bundle]['group_hierarchy'][$group_name],
            ];
          }
          // Check if the child is another group (nested groups).
          elseif (isset($field_groups[$child_name]) && isset($groups[$child_name])) {
            $this->logger->info('Found nested group @child in group @parent for bundle @bundle', [
              '@child' => $child_name,
              '@parent' => $group_name,
              '@bundle' => $bundle,
            ]);
          }
        }
      }
    }

    return $bundle_data;
  }

  /**
   * Remove all field groups from a form display.
   *
   * @param \Drupal\Core\Entity\Display\EntityFormDisplayInterface $form_display
   *   The form display to remove field groups from.
   */
  private function removeFieldGroupsFromFormDisplay(EntityFormDisplayInterface $form_display): void {
    if (!$this->moduleHandler->moduleExists('field_group')) {
      return;
    }

    $third_party_settings = $form_display->getThirdPartySettings('field_group');
    if (!empty($third_party_settings)) {
      // Remove each field group individually.
      foreach (array_keys($third_party_settings) as $group_name) {
        $form_display->unsetThirdPartySetting('field_group', $group_name);
        // Also remove the group component if it exists.
        $form_display->removeComponent($group_name);
      }
    }
  }

  /**
   * Migrates field storages for all unique fields across bundles.
   *
   * @param array $bundle_data
   *   Organized bundle data.
   * @param \Drupal\field\FieldStorageConfigStorage $field_storage_config_storage
   *   Field storage config storage service.
   * @param string $source_entity_type
   *   Source entity type.
   * @param string $target_entity_type
   *   Target entity type.
   *
   * @return array
   *   Array of migrated field names and their types.
   */
  private function migrateFieldStorage(array $bundle_data, FieldStorageConfigStorage $field_storage_config_storage, string $source_entity_type, string $target_entity_type): array {
    $migrated_fields = [];
    $unique_fields = [];

    // Collect all unique fields across bundles.
    foreach ($bundle_data as $bundle => $data) {
      foreach ($data['fields'] as $field_name => $field_info) {
        $unique_fields[$field_name] = $field_info;
      }
    }

    foreach ($unique_fields as $field_name => $field_info) {
      // Skip if field storage already exists.
      if ($field_storage_config_storage->load($target_entity_type . '.' . $field_name)) {
        continue;
      }

      $field_storage_config = $field_storage_config_storage->load($source_entity_type . '.' . $field_name);
      if (!$field_storage_config instanceof FieldStorageConfigInterface) {
        // Source field storage not found; skip.
        continue;
      }

      try {
        // Create the new field storage based on the source storage, ensuring
        // we keep the same language as the existing field storage config.
        $new_field_storage = $field_storage_config_storage->create([
          'field_name' => $field_name,
          'entity_type' => $target_entity_type,
          'type' => $field_storage_config->getType(),
          'cardinality' => $field_storage_config->getCardinality(),
          'settings' => $field_storage_config->getSettings(),
          // Preserve whether the field is translatable so values can have
          // translations just like in the source storage.
          'translatable' => $field_storage_config->isTranslatable(),
          // Preserve the configuration language if available on source.
          // Config entities are translatable; use the source langcode so the
          // base translation matches the original.
          'langcode' => $field_storage_config->language()->getId(),
        ]);
        $new_field_storage->save();

        // Copy translations for the FieldConfig label/description and
        // settings if any.
        $this->migrateFieldConfigTranslations($field_storage_config, $new_field_storage, 'field.storage.');

        $migrated_fields[$field_name] = $field_storage_config->getType();
      }
      catch (\Exception $e) {
        $this->logger->error('Error creating field storage @field: @error', [
          '@field' => $field_name,
          '@error' => $e->getMessage(),
        ]);
      }
    }

    return $migrated_fields;
  }

  /**
   * Finds a target bundle by traversing group hierarchy and checking labels.
   *
   * @param string $group_name
   *   The name of the group.
   * @param array $all_groups
   *   All available field groups.
   * @param array $target_bundles
   *   Available target bundle machine names.
   *
   * @return string|null
   *   The determined target bundle or NULL if not found.
   */
  private function findBundleByLabelHierarchy(string $group_name, array $all_groups, array $target_bundles): ?string {
    $current_group_name = $group_name;
    // Prevent infinite loops.
    $max_depth = 10;
    $depth = 0;

    // Traverse up the hierarchy to find a matching label.
    while ($current_group_name && isset($all_groups[$current_group_name]) && $depth < $max_depth) {
      $current_group = $all_groups[$current_group_name];
      $current_label = $current_group->label;

      // Convert label to machine name and check if it matches a bundle.
      $converted_bundle_name = $this->convertLabelByMachineName($current_label);
      if (in_array($converted_bundle_name, $target_bundles)) {
        return $converted_bundle_name;
      }

      // Move to the parent group.
      $current_group_name = !empty($current_group->parent_name) ? $current_group->parent_name : NULL;
      $depth++;
    }

    return NULL;
  }

  /**
   * Recreates field groups for a specific bundle with proper hierarchy.
   *
   * @param string $entity_type
   *   Target entity type.
   * @param string $bundle
   *   Target bundle.
   * @param array $groups
   *   Field groups to recreate.
   * @param array $fields
   *   Available fields for this bundle.
   * @param array $group_hierarchy
   *   Group hierarchy information.
   */
  private function recreateFieldGroupsByBundle(string $entity_type, string $bundle, array $groups, array $fields, array $group_hierarchy): void {
    // Sort groups by hierarchy depth (parents first).
    $sorted_groups = $this->sortGroupsByHierarchy($groups);

    foreach ($sorted_groups as $group_name => $group_info) {
      // Filter children to only include fields and groups that exist in this
      // bundle.
      $migrated_children = [];
      foreach ($group_info->children as $child) {
        if (isset($fields[$child])) {
          $migrated_children[] = $child;
        }
        elseif (isset($groups[$child])) {
          // Child is another group, include it.
          $migrated_children[] = $child;
        }
      }

      // Skip empty groups.
      if (empty($migrated_children)) {
        continue;
      }

      // Validate parent_name exists in our groups and update parent reference.
      $parent_name = $group_info->parent_name;
      if ($parent_name && !isset($groups[$parent_name])) {
        // If the parent was an ignored group, set as root level.
        $parent_name = '';
      }

      $new_group = (object) [
        'group_name' => $group_info->group_name,
        'entity_type' => $entity_type,
        'bundle' => $bundle,
        'mode' => 'default',
        'context' => 'form',
        'children' => $migrated_children,
        'parent_name' => $parent_name,
        'weight' => $group_info->weight ?? 0,
        'label' => $group_info->label,
        'format_type' => $group_info->format_type ?? 'tab',
        'format_settings' => $group_info->format_settings ?? [],
        'region' => $group_info->region ?? 'content',
      ];

      try {
        field_group_group_save($new_group);

        $this->logger->info('Created field group @group (label: "@label") for bundle @bundle with children: @children', [
          '@group' => $group_name,
          '@label' => $group_info->label,
          '@bundle' => $bundle,
          '@children' => implode(', ', $migrated_children),
        ]);
      }
      catch (\Exception $e) {
        $this->logger->error('Exception creating field group @group for bundle @bundle: @error', [
          '@group' => $group_name,
          '@bundle' => $bundle,
          '@error' => $e->getMessage(),
        ]);
      }
    }
  }

  /**
   * Sorts groups by hierarchy (parents first).
   *
   * @param array $groups
   *   Field groups to sort.
   *
   * @return array
   *   Sorted groups array.
   */
  private function sortGroupsByHierarchy(array $groups): array {
    $sorted = [];
    $processed = [];

    // Recursive function to add groups in proper order.
    $add_group = function ($group_name) use ($groups, &$sorted, &$processed, &$add_group) {
      if (isset($processed[$group_name]) || !isset($groups[$group_name])) {
        return;
      }

      $group = $groups[$group_name];

      // Add a parent first if it exists and is in our groups.
      if (!empty($group->parent_name) && isset($groups[$group->parent_name])) {
        $add_group($group->parent_name);
      }

      // Add this group.
      $sorted[$group_name] = $group;
      $processed[$group_name] = TRUE;
    };

    // Process all groups.
    foreach ($groups as $group_name => $group_info) {
      $add_group($group_name);
    }

    return $sorted;
  }

  /**
   * Extracts field data for a specific bundle from a node.
   *
   * @param \Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface $entity_splitter_plugin
   *   The entity splitter plugin.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The content entity interface.
   * @param string $bundle
   *   The bundle identifier.
   *
   * @return array
   *   Array of field data keyed by the field name.
   */
  private function extractFieldDataForBundle(EntitySplitterPluginInterface $entity_splitter_plugin, ContentEntityInterface $entity, string $bundle): array {
    $field_data = [];

    // Get field definitions for the bundle.
    $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_splitter_plugin->getTargetEntityTypeId(), $bundle);

    foreach ($field_definitions as $field_name => $field_definition) {
      // Skip base fields and fields that don't exist on the node.
      if (!$field_definition instanceof FieldConfig || !$entity->hasField($field_name)) {
        continue;
      }

      // Get field value from node if it exists and is not empty.
      if (!$entity->get($field_name)->isEmpty()) {
        $field_data[$field_name] = $entity->get($field_name)->getValue();
      }
    }

    return $field_data;
  }

  /**
   * Builds the hierarchy path for a group, filtering by labels.
   *
   * @param string $group_name
   *   The name of the group.
   * @param array $all_groups
   *   All available field groups.
   *
   * @return array
   *   Array representing the hierarchy path from root to the given group.
   */
  private function buildGroupHierarchyPathByLabels(string $group_name, array $all_groups): array {
    $path = [];
    $current_group_name = $group_name;
    // Prevent infinite loops.
    $max_depth = 10;
    $depth = 0;

    // Build a path from the current group back to the root.
    while ($current_group_name && isset($all_groups[$current_group_name]) && $depth < $max_depth) {
      $current_group = $all_groups[$current_group_name];

      // Move to the parent group.
      $current_group_name = !empty($current_group->parent_name) ? $current_group->parent_name : NULL;
      $depth++;
    }

    return $path;
  }

}
