<?php

declare(strict_types=1);

namespace Drupal\entity_splitter\Commands;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_splitter\EntitySplitterActions;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;

/**
 * The entity migration commands.
 */
class EntitySplitterCommands extends DrushCommands {
  use StringTranslationTrait;

  public function __construct(protected EntitySplitterActions $actions) {
    parent::__construct();
  }

  /**
   * The entity splitter migrate structure command.
   */
  #[CLI\Command(name: 'entity-splitter:ms', aliases: ['es:ms'])]
  #[CLI\Help(description: 'Run an entity splitter plugin')]
  #[CLI\Argument(name: 'plugin_id', description: 'The entity splitter plugin ID')]
  #[CLI\Option(name: 'mapping-file', description: 'Path to YAML mapping file')]
  #[CLI\Option(name: 'use-field-groups', description: 'Build mapping from field groups')]
  #[CLI\Option(name: 'create-fields', description: 'If fields (config) should be created')]
  #[CLI\Option(name: 'create-reference-fields', description: 'If reference fields (config) should be created in source entity.')]
  #[CLI\Option(name: 'hide-source-fields', description: 'Option to hide the mapped source fields.')]
  #[CLI\Option(name: 'remove-source-fields', description: 'Option to remove the mapped source fields (Be careful you will lose all the data).')]
  #[CLI\Option(name: 'remove-source-field-groups', description: 'Option to remove the source field groups.')]
  #[CLI\Option(name: 'create-new-target-form-modes', description: 'Create new form modes for the target entity types (this is handy for inline entity form).')]
  #[CLI\Option(name: 'form-mode', description: 'form mode used for the field groups (this is only when use-field-groups isset).')]
  #[CLI\Option(name: 'dry-run', description: 'Do not save changes')]
  public function migrateStructure(
    string $plugin_id,
    array $options = [
      'mapping-file' => NULL,
      'use-field-groups' => TRUE,
      'create-fields' => FALSE,
      'create-reference-fields' => FALSE,
      'hide-source-fields' => FALSE,
      'remove-source-fields' => FALSE,
      'remove-source-field-groups' => FALSE,
      'create-new-target-form-modes' => '',
      'dry-run' => FALSE,
      'form-mode' => 'default',
    ],
  ): int {
    $result = $this->actions->migrateStructure($plugin_id, $options);
    $this->io()->success($result->getSummary());

    return $result->getFailed() > 0 ? self::EXIT_FAILURE : self::EXIT_SUCCESS;
  }

  /**
   * The entity splitter migrate data command.
   */
  #[CLI\Command(name: 'entity-splitter:md', aliases: ['es:md'])]
  #[CLI\Help(description: 'Run an entity splitter migration')]
  #[CLI\Argument(name: 'plugin_id', description: 'The entity splitter plugin ID')]
  #[CLI\Option(name: 'process-all', description: 'If we should process all the items.')]
  #[CLI\Option(name: 'use-queue', description: 'If the migration should use a queue.')]
  #[CLI\Option(name: 'entity-id', description: 'The ID of the entity to migrate the data for.')]
  #[CLI\Option(name: 'remove-existing-references', description: 'If we should remove existing sub-entity references while migrating entities.')]
  public function migrateData(
    string $plugin_id,
    array $options = [
      'mapping-file' => NULL,
      'process-all' => TRUE,
      'use-queue' => FALSE,
      'entity-id' => NULL,
      'remove-existing-references' => FALSE,
    ],
  ): int {
    $sandbox = [];
    $result = $this->actions->migrateData($plugin_id, $sandbox, $options);
    $this->io()->success($result);
    return self::EXIT_SUCCESS;
  }

  /**
   * The entity migration list command.
   */
  #[CLI\Command(name: 'entity-splitter:list', aliases: ['es:list'])]
  #[CLI\Help(description: 'List available entity splitter plugin IDs')]
  public function listPlugins(): int {
    $definitions = $this->actions->getDefinitions();
    if (empty($definitions)) {
      $this->io()->warning('No entity splitter plugins found.');
      return self::EXIT_SUCCESS;
    }
    $rows = [];
    foreach ($definitions as $id => $def) {
      $rows[] = [
        $id,
        (string) ($def['label'] ?? $id),
        (string) ($def['source_entity_type'] ?? ''),
        (string) ($def['source_bundle'] ?? ''),
        (string) ($def['description'] ?? ''),
      ];
    }
    $this->io()->table(['ID', 'Label', 'Source entity', 'Source bundle', 'Description'], $rows);
    return self::EXIT_SUCCESS;
  }

}
