<?php

declare(strict_types=1);

namespace Drupal\entity_splitter;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\entity_splitter\Mapping\FieldGroupMapper;
use Drupal\entity_splitter\Mapping\YamlMappingReader;
use Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface;
use Drupal\shs\StringTranslationTrait;

/**
 * Service that executes Entity Migration plugins.
 */
class EntitySplitterActions {

  use StringTranslationTrait;

  public function __construct(
    private EntitySplitterManager $pluginManager,
    private FileSystemInterface $fileSystem,
    private YamlMappingReader $yamlReader,
    private FieldGroupMapper $fieldGroupMapper,
    private LoggerChannelInterface $logger,
  ) {}

  /**
   * Runs a plugin which handles actions like creating fields and references.
   *
   * @param string $plugin_id
   *   The plugin ID.
   * @param array $options
   *   Options like:
   *   - mapping_file: string path to YAML mapping file.
   *   - use_field_groups: bool to derive mapping from field_group config.
   */
  public function migrateStructure(string $plugin_id, array $options = []): EntitySplitterResult {
    $options = $this->prepareOptions($options);
    $plugin = $this->getPlugin($plugin_id);
    if (!$plugin) {
      return new EntitySplitterResult($plugin_id, failed: 1, messages: [$this->t('Could not instantiate migration plugin @id.', ['@id' => $plugin_id])]);
    }

    // Load mapping from YAML or Field Groups if a plugin didn't receive one.
    if (!isset($options['mapping'])) {
      if (!empty($options['mapping-file'])) {
        $options['mapping'] = $this->yamlReader->read((string) $options['mapping-file']);
      }
      elseif (!empty($options['use-field-groups'])) {
        $options['mapping'] = $this->fieldGroupMapper->buildMapping($plugin, $options['form-mode'] ?? 'default');
      }
    }

    // Run the plugin.
    $result = $plugin->migrateStructure($options);
    $this->logger->info($result->getSummary());
    return $result;
  }

  /**
   * Migrate entities, either directly or queue.
   *
   * @param string $plugin_id
   *   The plugin id.
   * @param array $sandbox
   *   The update hook sandbox.
   * @param array $options
   *   The options (use-queue).
   *
   * @return string|\Drupal\Core\StringTranslation\TranslatableMarkup
   *   The migration result.
   */
  public function migrateData(string $plugin_id, array &$sandbox, array $options = ['use-queue' => FALSE]): string|TranslatableMarkup {
    $plugin = $this->getPlugin($plugin_id);
    if (!$plugin) {
      return '';
    }

    // Migrate the data batched or all.
    return $plugin->migrateData($sandbox, $options);
  }

  /**
   * Get entity splitter plugin instance.
   *
   * @param string $plugin_id
   *   The plugin id.
   *
   * @return \Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface|null
   *   Returns the plugin instance or NULL if it could not be instantiated.
   */
  public function getPlugin(string $plugin_id): ?EntitySplitterPluginInterface {
    try {
      /** @var \Drupal\entity_splitter\Plugin\EntitySplitterPluginInterface $plugin */
      $plugin = $this->pluginManager->createInstance($plugin_id);
    }
    catch (PluginException $e) {
      $this->logger->error('Could not instantiate migration plugin @id: @msg', [
        '@id' => $plugin_id,
        '@msg' => $e->getMessage(),
      ]);
      return NULL;
    }
    return $plugin;
  }

  /**
   * Returns plugin definitions (ID => definition array).
   */
  public function getDefinitions(): array {
    return $this->pluginManager->getDefinitions();
  }

  /**
   * Prepares options with defaults and resolves file paths.
   */
  private function prepareOptions(array $options): array {
    $options['dry_run'] = (bool) ($options['dry_run'] ?? FALSE);

    if (!empty($options['mapping_file'])) {
      $path = (string) $options['mapping_file'];
      $real = $this->fileSystem->realpath($path) ?: $path;
      $options['mapping_file'] = $real;
    }
    return $options;
  }

}
