<?php

namespace Drupal\entity_splitter;

/**
 * Value object representing the result of a migration run.
 */
class EntitySplitterResult {

  /**
   * Constructs an EntitySplitterResult object.
   */
  public function __construct(
    protected string $pluginId,
    private int $processed = 0,
    private int $created = 0,
    private int $updated = 0,
    private int $failed = 0,
    private mixed $messages = [],
    private bool $dryRun = FALSE,
  ) {}

  /**
   * Returns a printable summary string.
   *
   * @return string
   *   The summary.
   */
  public function getSummary(): string {
    return sprintf(
      'Plugin %s: processed=%d, created=%d, updated=%d, failed=%d%s',
      $this->pluginId,
      $this->processed,
      $this->created,
      $this->updated,
      $this->failed,
      $this->dryRun ? ' (dry-run)' : ''
    );
  }

  /**
   * Converts the result to an array.
   *
   * @return array
   *   Returns the result as an array.
   */
  public function toArray(): array {
    return [
      'plugin' => $this->pluginId,
      'processed' => $this->processed,
      'created' => $this->created,
      'updated' => $this->updated,
      'failed' => $this->failed,
      'messages' => $this->messages,
      'dry_run' => $this->dryRun,
    ];
  }

  /**
   * Gets the plugin id.
   */
  public function getPluginId(): string {
    return $this->pluginId;
  }

  /**
   * Gets processed count.
   */
  public function getProcessed(): int {
    return $this->processed;
  }

  /**
   * Gets created count.
   */
  public function getCreated(): int {
    return $this->created;
  }

  /**
   * Gets updated count.
   */
  public function getUpdated(): int {
    return $this->updated;
  }

  /**
   * Gets failed count.
   */
  public function getFailed(): int {
    return $this->failed;
  }

  /**
   * Gets messages.
   *
   * @return string
   *   The messages as a string.
   */
  public function getMessages(): string {
    return is_array($this->messages) ? implode('; ', $this->messages) : (string) $this->messages;
  }

  /**
   * Whether this run was a dry-run.
   */
  public function isDryRun(): bool {
    return $this->dryRun;
  }

}
