<?php

namespace Drupal\entity_splitter\Plugin\QueueWorker;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\entity_splitter\EntitySplitterActions;
use Drupal\entity_splitter\Processing\EntitySplitterProcessor;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Handles the data migration inside a queue worker.
 */
#[QueueWorker(
  id: 'entity_splitter_field_data_migration',
  title: new TranslatableMarkup('The entity splitter data migration queue'),
  cron: ['time' => 45]
)]
class EntitySplitterDataMigrationQueue extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  use LoggerChannelTrait;

  /**
   * The delay interval equal to 15 minutes.
   */
  const int DELAY_INTERVAL = 900;

  /**
   * The logger interface.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a new class instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager service.
   * @param \Drupal\entity_splitter\EntitySplitterActions $entitySplitterActions
   *   Entity splitter actions.
   * @param \Drupal\entity_splitter\Processing\EntitySplitterProcessor $entitySplitterProcessor
   *   Entity splitter processor.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntitySplitterActions $entitySplitterActions,
    protected EntitySplitterProcessor $entitySplitterProcessor,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->logger = $this->getLogger('entity_splitter');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('entity_splitter.actions'),
      $container->get('entity_splitter.processor')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    if (empty($data['entity_id'])) {
      $this->logger->error('No Entity ID found in queue item');
      return;
    }

    if (empty($data['plugin_id'])) {
      $this->logger->error('No entity splitter plugin id found in queue item');
      return;
    }

    // Initialise the required variables.
    $plugin_id = $data['plugin_id'];
    $entity_id = $data['entity_id'];

    // Get the entity splitter plugin instance.
    $entity_splitter_plugin = $this->entitySplitterActions->getPlugin($plugin_id);
    if (!$entity_splitter_plugin) {
      $this->logger->error('Could not find entity splitter plugin with id @id', ['@id' => $plugin_id]);
      return;
    }

    $entity = $this->entityTypeManager->getStorage($entity_splitter_plugin->getSourceEntityTypeId())->load($entity_id);

    // Check if the entity is a content entity.
    if (!$entity instanceof ContentEntityInterface) {
      $this->logger->error('Entity with id @id is not a content entity', ['@id' => $entity_id]);
      return;
    }

    // Only process the latest revision or published entity.
    if (!$entity->isDefaultRevision() || ($entity instanceof EntityPublishedInterface && !$entity->isPublished())) {
      $this->logger->error('Entity with id @id is not the latest revision or the published entity', ['@id' => $entity_id]);
      return;
    }

    // Migrate the entity data.
    $this->entitySplitterProcessor->migrateFieldData($entity_splitter_plugin, [
      'current_entity' => $entity_id,
      'remove-existing-references' => $data['remove-existing-references'] ?? FALSE,
    ]);

  }

}
