<?php

namespace Drupal\entity_translate_side_by_side\Hook;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Url;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Hook\Attribute\Hook;



final readonly class EntityTranslateSideBySideHooks {

  /**
   * Constructs a new instance of the class.
   *
   * @param ModuleHandlerInterface $moduleHandler
   *   The module handler service is used to interact with modules.
   */
  public function __construct(
    private ModuleHandlerInterface $moduleHandler,
  ) {}

  /**
   * Implements hook_entity_operation().
   *
   * Adds a "Translate side by side" operation for entities with numeric IDs.
   * This operation links to a custom route for side-by-side translation.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to which the operation is being added.
   *
   * @return array<string, mixed>
   *   An array of operations.
   */
  #[Hook('entity_operation')]
  public function addTranslateSideBySideOperation(EntityInterface $entity): array {
    /**
     * Hardcoded lists of entities that cannot yet be translated and lead to an exception.
     */
    $skip = [
      /* TODO for menu_link_content: Entity type is currently not translatable. Implementation required.
       *  Issue: https://www.drupal.org/project/entity_translate_side_by_side/issues/3441547
       */
      'menu_link_content',
    ];

    // Give other modules a chance to alter and influence the skip list.
    $this->moduleHandler->alter('entity_translate_side_by_side_skip', $skip);

    $operations = [];
    /* TODO: Make entity types for which entity operations are to be offered configurable via a user interface.
     *  Issue: https://www.drupal.org/project/entity_translate_side_by_side/issues/3416987
     */
    if ($entity instanceof TranslatableInterface &&
      is_numeric($entity->id()) &&
      $entity->isTranslatable() &&
      !\in_array($entity->getEntityTypeId(), $skip, TRUE)) {
      $operations['translate_side_by_side'] = [
        'title' => t('Translate side by side'),
        'url' => Url::fromRoute(
          'entity_translate_side_by_side',
          [
            'entity_type' => $entity->getEntityTypeId(),
            'entity_id' => $entity->id(),
          ],
          ['query' => ['destination' => Url::fromRoute('<current>')->toString()]]
        ),
        'weight' => 10,
      ];
    }

    return $operations;
  }
}
