<?php

namespace Drupal\Tests\entity_translation_sync\Functional;

use Drupal\Core\Entity\EntityInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Tests\content_translation\Functional\ContentTranslationTestBase;
use Drupal\user\UserInterface;

/**
 * Base functional tests class.
 *
 * It provides:
 *   - Testing fields
 *   - Module basic and extensible configuration.
 *   - User able to sync anything.
 */
abstract class EntityTranslationSyncTestsBase extends ContentTranslationTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'entity_test',
    'language',
    'content_translation',
    'entity_translation_sync',
    'entity_translation_sync_test',
  ];

  /**
   * User able to synchronize everything.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $synchronizerFullUser;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected $entityTypeId = 'entity_test_mul';

  /**
   * {@inheritdoc}
   */
  protected $fieldName = 'field_etsync_text';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->doSetup();
    // Needed to rebuild link templates and routes.
    $this->rebuildAll();
    $this->createTestData();
  }

  /**
   * {@inheritdoc}
   */
  protected function setupTestFields() {
    parent::setupTestFields();
    $text_field_names = [
      'field_etsync_text_2' => 'Test translatable text-field 2',
      'field_etsync_not' => 'Test not synchronizable text-field',
      'field_etsync_restricted_access' => 'Test synchronizable text-field with restricted access',
    ];

    foreach ($text_field_names as $field_name => $field_label) {
      FieldStorageConfig::create([
        'field_name' => $field_name,
        'type' => 'string',
        'entity_type' => $this->entityTypeId,
        'cardinality' => 1,
        'translatable' => 1,
      ])->save();
      FieldConfig::create([
        'entity_type' => $this->entityTypeId,
        'field_name' => $field_name,
        'bundle' => $this->bundle,
        'label' => $field_label,
        'translatable' => 1,
      ])->save();
      /** @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface $display_repository */
      $display_repository = \Drupal::service('entity_display.repository');
      $display_repository->getFormDisplay($this->entityTypeId, $this->bundle, 'default')
        ->setComponent($field_name, [
          'type' => 'string_textfield',
          'weight' => 0,
        ])
        ->save();
    }

  }

  /**
   * {@inheritdoc}
   */
  protected function enableTranslation() {
    parent::enableTranslation();
    $this->setupEntityTranslationSync();
  }

  /**
   * Setup module configuration.
   */
  protected function setupEntityTranslationSync() {
    $this->config('entity_translation_sync.settings')
      ->set('entity_types', $this->getConfiguredEntityTypes())
      ->save();
  }

  /**
   * List of entity types configured to be synchronizable.
   *
   * @return array
   *   By default, the test entity with the test fields.
   */
  protected function getConfiguredEntityTypes() : array {
    return [
      $this->entityTypeId => [
        'bundles' => [
          $this->bundle => [
            'fields' => [
              'field_etsync_text',
              'field_etsync_text_2',
              'field_etsync_restricted_access',
            ],
          ],
        ],
      ],
    ];
  }

  /**
   * Creates and activates translator, editor and admin users.
   */
  protected function setupUsers() {
    $this->synchronizerFullUser = $this->drupalCreateUser(
      array_merge(
        $this->getEntityTestViewPermissions(),
        $this->getTranslatorPermissions(), [
          'synchronize any entity translation',
          'edit field_etsync_restricted_access field',
        ]),
      'synchronizer_full',
    );
  }

  /**
   * Get permissions to view the test entity.
   *
   * It is needed as being able to synchronize the entity implies
   * being able to view it.
   *
   * @return array
   *   Permissions to view entity + translations.
   */
  protected function getEntityTestViewPermissions() {
    return [
      'view test entity',
      'view test entity translations',
    ];
  }

  /**
   * Get an entity from the database based on its label.
   *
   * @param string|\Drupal\Component\Render\MarkupInterface $name
   *   An entity label, usually generated by $this->randomMachineName().
   * @param bool $reset
   *   (optional) Whether to reset the entity cache.
   *
   * @throw \InvalidArgumentException
   *   When the entity does not exist.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface
   *   An entity matching $title.
   */
  protected function drupalGetEntityTestByLabel($name, $reset = FALSE) {
    if ($reset) {
      \Drupal::entityTypeManager()->getStorage($this->entityTypeId)->resetCache();
    }
    // Cast MarkupInterface objects to string.
    $entity_type = \Drupal::entityTypeManager()->getDefinition($this->entityTypeId);
    $name = (string) $name;
    $entities = \Drupal::entityTypeManager()
      ->getStorage($this->entityTypeId)
      ->loadByProperties([$entity_type->getKey('label') => $name]);
    // Load the first node returned from the database.
    $returned_entity = reset($entities);
    if (!$returned_entity instanceof EntityInterface) {
      throw new \InvalidArgumentException(sprintf('Entity with label "%s" does not exist.', $name));
    }
    return $returned_entity;
  }

  /**
   * Create entities to do the tests.
   */
  protected function createTestData() {

    $this->createEntity([
      'name' => 'Test entity with sync access',
      'field_etsync_text' => 'Etsync text',
      'field_etsync_text_2' => 'Etsync text 2',
      'field_etsync_not' => 'Etsync text not',
      'field_etsync_restricted_access' => 'Etsync restricted access',
    ], 'en');

    $this->createEntity([
      'name' => 'Test entity without sync access',
      'field_etsync_text' => $this->randomString(),
      'field_etsync_text_2' => $this->randomString(),
      'field_etsync_not' => $this->randomString(),
      'field_etsync_restricted_access' => 'Etsync restricted access',
    ], 'en', 'other_bundle');
  }

}
