<?php

namespace Drupal\entity_translation_unified_form\Plugin\EntityTranslationUnifiedFormMode;

use Drupal\Core\Render\Element;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_translation_unified_form\EntityTranslationUnifiedFormModeInterface;

/**
 * Provides an inline translation mode for entity forms.
 *
 * @EntityTranslationUnifiedFormMode (
 *   id = "EntityTranslationUnifiedFormInlineMode",
 *   admin_label = @Translation("Inline Mode"),
 * )
 */
class EntityTranslationUnifiedFormInlineMode implements EntityTranslationUnifiedFormModeInterface {
  use StringTranslationTrait;

  /**
   * Alters the form fields for translation.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param array &$field
   *   The field definition being altered.
   * @param string $field_name
   *   The name of the field.
   * @param \Drupal\Core\Language\LanguageInterface $language
   *   The language object.
   */
  public function fieldFormAlter($form, $form_state, &$field, $field_name, $language) {
    $this->alterTitle($form, $form_state, $field, $field_name, $language);
  }

  /**
   * Returns the theme wrapper for the field group.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param array $field
   *   The field definition.
   * @param string $field_name
   *   The name of the field.
   *
   * @return string
   *   The theme wrapper name.
   */
  public function getFieldGroupThemeWrapper($form, $form_state, $field, $field_name) {
    return 'entity_translation_unified_form__inline__wrapper';
  }

  /**
   * Returns the theme wrapper for a field.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param array $field
   *   The field definition.
   * @param string $field_name
   *   The name of the field.
   * @param \Drupal\Core\Language\LanguageInterface $language
   *   The language object.
   *
   * @return string
   *   The theme wrapper name.
   */
  public function getFieldThemeWrapper($form, $form_state, $field, $field_name, $language) {
    return 'entity_translation_unified_form__inline__field_wrapper';
  }

  /**
   * Adds the language label to the field title.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param array &$field
   *   The field definition.
   * @param string $field_name
   *   The name of the field.
   * @param \Drupal\Core\Language\LanguageInterface $language
   *   The language object.
   */
  protected function alterTitle($form, $form_state, &$field, $field_name, $language) {
    if (!isset($field['widget']) || !is_object($language)) {
      return;
    }

    // Get language display mode.
    $form_object = $form_state->getFormObject();
    $entity = $form_object->getEntity();
    $entity_type_id = $entity->getEntityTypeId();
    $bundle = $entity->bundle();
    $language_display = entity_translation_unified_form_language_display($entity_type_id, $bundle);

    if ($language_display == 'native') {
      // Get native languages.
      $language_manager = \Drupal::languageManager();
      $native_languages = $language_manager->getNativeLanguages();
      $current_language = $native_languages[$language->getId()];
      // Get language name in its own language.
      $text = ' (' . $this->t('@language_name', ['@language_name' => $current_language->getName()], ['langcode' => $language->getId()]) . ')';
    }
    elseif ($language_display == 'current') {
      $text = ' (' . $this->t('@language_name', ['@language_name' => $language->getName()]) . ')';
    }
    else {
      $text = ' (' . $language->getId() . ')';
    }

    $widget = &$field['widget'];
    $this->addTranslatabilityClue($widget, $text);
  }

  /**
   * Adds a clue about the form element's translatability.
   *
   * If the given element does not have a #title attribute, the function is
   * recursively applied to child elements.
   *
   * @param array $element
   *   A form element array.
   * @param string $suffix
   *   The text to append to the title.
   */
  protected function addTranslatabilityClue(&$element, $suffix) {
    static $fapi_title_elements;
    $done = TRUE;

    // Elements that can have a #title attribute according to FAPI Reference.
    if (!isset($fapi_title_elements)) {
      $fapi_title_elements = array_flip([
        'managed_file', 'checkbox', 'checkboxes', 'date', 'details', 'fieldset',
        'file', 'item', 'password', 'password_confirm', 'radio', 'radios',
        'select', 'text_format', 'textarea', 'textfield', 'weight',
      ]);
    }

    // Update #title for valid elements.
    if (isset($element['#type']) && isset($fapi_title_elements[$element['#type']]) && isset($element['#title'])) {
      $element['#title'] .= $suffix;

      // If the title is invisible, mark as not done.
      if (isset($element['#title_display']) && ($element['#title_display'] == 'invisible')) {
        return FALSE;
      }
      else {
        return TRUE;
      }
    }

    // Apply to child elements if no valid title exists.
    if ($children = Element::children($element)) {
      foreach ($children as $delta) {
        if (!$this->addTranslatabilityClue($element[$delta], $suffix)) {
          $done = FALSE;
        }
      }
    }

    // If children are unchanged, apply to parent.
    if (!$done && isset($element['#title'])) {
      $element['#title'] .= $suffix;
    }
  }

}
