<?php

declare(strict_types=1);

namespace Drupal\entity_type_behaviors\Plugin\GraphQL\DataProducer;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\entity_type_behaviors\Helper\EntityTypeBehaviorHelper;
use Drupal\entity_type_behaviors\Config\BehaviorConfigFactory;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Get entity type behaviors for an entity.
 *
 * @DataProducer(
 *   id = "entity_type_behaviors",
 *   name = @Translation("Entity Type behaviors"),
 *   description = @Translation("Entity Type behaviors information for entity."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Entity Type behaviors settings")
 *   ),
 *   consumes = {
 *     "entity" = @ContextDefinition("entity",
 *       label = @Translation("Entity"),
 *     ),
 *   }
 * )
 */
class EntityTypeBehaviors extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  private LoggerInterface $logger;

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    private readonly EntityTypeBehaviorHelper $entityTypeBehaviorHelper,
    private readonly BehaviorConfigFactory $behaviorConfigFactory,
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->logger = $loggerFactory->get('entity_type_behaviors');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type_behaviors.helper.entity_type_behavior'),
      $container->get('entity_type_behaviors.config.factory'),
      $container->get('logger.factory')
    );
  }

  /**
   * Return the entity behaviors.
   */
  public function resolve(?ContentEntityInterface $entity): ?array {
    if (!$entity) {
      return NULL;
    }

    try {
      $entityTypeId = $entity->getEntityTypeId();
      $bundleId = $entity->bundle();

      // Get enabled behaviors for this entity type/bundle.
      $enabledBehaviors = $this->behaviorConfigFactory->getEnabledConfiguredBehaviors($entityTypeId, $bundleId);
      if (empty($enabledBehaviors)) {
        return NULL;
      }

      // Get behavior values from the entity.
      $behaviorValues = $this->entityTypeBehaviorHelper->getBehaviorValuesForEntity($entity);
      if (empty($behaviorValues)) {
        return NULL;
      }

      return $this->formatBehaviorValues($behaviorValues, $enabledBehaviors);
    }
    catch (\Throwable $e) {
      $this->logger->error('Error resolving behaviors: @error', [
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Format behavior values for GraphQL.
   */
  private function formatBehaviorValues(array $behaviorValues, array $enabledBehaviors): ?array {
    $result = [];

    foreach ($behaviorValues as $behaviorId => $values) {
      // Only include behaviors that are enabled for this entity type/bundle.
      if (!isset($enabledBehaviors[$behaviorId])) {
        continue;
      }

      // Add the behavior type for the interface.
      $values['behaviorType'] = $behaviorId;

      // Add this behavior to the result.
      $result[] = $values;
    }

    return !empty($result) ? $result : NULL;
  }

}
