<?php

declare(strict_types=1);

namespace Drupal\entity_type_behaviors\Plugin\GraphQL\SchemaExtension;

use Drupal\graphql\GraphQL\ResolverBuilder;
use Drupal\graphql\GraphQL\ResolverRegistryInterface;
use Drupal\graphql_compose\Plugin\GraphQL\SchemaExtension\ResolverOnlySchemaExtensionPluginBase;
use Drupal\entity_type_behaviors\Config\BehaviorConfigFactory;
use Drupal\entity_type_behaviors\EntityTypeBehaviorManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Entity type behaviors GraphQL Extension.
 *
 * @SchemaExtension(
 *   id = "entity_type_behaviors_schema_extension",
 *   name = "GraphQL Compose Entity Type Behaviors",
 *   description = "Entity type behaviors",
 *   schema = "graphql_compose"
 * )
 */
class EntityTypeBehaviorsSchemaExtension extends ResolverOnlySchemaExtensionPluginBase {

  /**
   * The behavior configuration factory.
   *
   * @var \Drupal\entity_type_behaviors\Config\BehaviorConfigFactory
   */
  protected BehaviorConfigFactory $behaviorConfigFactory;

  /**
   * The entity type behavior manager.
   *
   * @var \Drupal\entity_type_behaviors\EntityTypeBehaviorManager
   */
  protected EntityTypeBehaviorManager $behaviorManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->behaviorConfigFactory = $container->get('entity_type_behaviors.config.factory');
    $instance->behaviorManager = $container->get('plugin.manager.entity_type_behavior');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function registerResolvers(ResolverRegistryInterface $registry): void {
    $builder = new ResolverBuilder();

    $this->registerEntityTypeBehaviorResolvers($registry, $builder);
    $this->registerBehaviorTypeResolvers($registry, $builder);
  }

  /**
   * Register entity type behavior resolvers.
   */
  private function registerEntityTypeBehaviorResolvers(ResolverRegistryInterface $registry, ResolverBuilder $builder): void {
    $entityTypeBundles = $this->behaviorConfigFactory->getConfiguredEntityTypesAndBundles();

    foreach ($entityTypeBundles as $entityTypeId => $bundles) {
      $plugin = $this->gqlEntityTypeManager->getPluginInstance($entityTypeId);
      if (!$plugin) {
        continue;
      }

      $bundleInfos = $plugin->getBundles();

      foreach ($bundles as $bundleId) {
        $enabledBehaviors = $this->behaviorConfigFactory->getEnabledConfiguredBehaviors($entityTypeId, $bundleId);
        if (empty($enabledBehaviors)) {
          continue;
        }

        $bundleTypeName = $this->findBundleTypeName($bundleInfos, $bundleId);
        if (!$bundleTypeName) {
          continue;
        }

        $registry->addFieldResolver(
          $bundleTypeName,
          'entity_type_behaviors',
          $builder->produce('entity_type_behaviors')
            ->map('entity', $builder->fromParent())
        );
      }
    }
  }

  /**
   * Find bundle type name.
   */
  private function findBundleTypeName(array $bundleInfos, string $bundleId): ?string {
    foreach ($bundleInfos as $bundleInfo) {
      if ($bundleInfo->getEntity()->id() === $bundleId) {
        return $bundleInfo->getTypeSdl();
      }
    }

    return NULL;
  }

  /**
   * Register behavior type resolvers.
   */
  private function registerBehaviorTypeResolvers(ResolverRegistryInterface $registry, ResolverBuilder $builder): void {
    // Register type resolvers for behavior union types.
    $entityTypesWithBehaviors = $this->getEntityTypesWithBehaviors();
    foreach ($entityTypesWithBehaviors as $entityTypeId => $bundles) {
      foreach ($bundles as $bundleId => $enabledBehaviors) {
        if (empty($enabledBehaviors)) {
          continue;
        }

        $unionTypeName = 'EntityTypeBehaviors' . $this->pascalCase($entityTypeId) . $this->pascalCase($bundleId);
        $this->registerUnionTypeResolver($registry, $unionTypeName);
      }
    }

    // Register field resolvers for behavior types.
    $this->registerBehaviorFieldResolvers($registry, $builder);
  }

  /**
   * Register union type resolver.
   */
  private function registerUnionTypeResolver(ResolverRegistryInterface $registry, string $unionTypeName): void {
    $registry->addTypeResolver(
      $unionTypeName,
      function ($value) {
        if (!isset($value['behaviorType'])) {
          return NULL;
        }

        return 'EntityTypeBehavior' . $this->pascalCase($value['behaviorType']);
      }
    );
  }

  /**
   * Register behavior field resolvers.
   */
  private function registerBehaviorFieldResolvers(ResolverRegistryInterface $registry, ResolverBuilder $builder): void {
    $definitions = $this->behaviorManager->getDefinitions();

    foreach ($definitions as $definition) {
      $behaviorId = $definition['id'];
      $typeName = 'EntityTypeBehavior' . $this->pascalCase($behaviorId);

      // Register behaviorType field resolver.
      $registry->addFieldResolver(
        $typeName,
        'behaviorType',
        $builder->callback(fn($value) => $behaviorId)
      );

      // Register field resolvers for each behavior field.
      $plugin = $this->behaviorManager->createInstance($behaviorId);
      $formElements = $plugin->getForm();

      foreach ($formElements as $fieldName => $formElement) {
        if (!isset($formElement['#type'])) {
          continue;
        }

        $registry->addFieldResolver(
          $typeName,
          $fieldName,
          $builder->callback(fn($value) => $value[$fieldName] ?? NULL)
        );
      }
    }
  }

  /**
   * Get entity types with behaviors.
   */
  private function getEntityTypesWithBehaviors(): array {
    $result = [];
    $entityTypeBundles = $this->behaviorConfigFactory->getConfiguredEntityTypesAndBundles();

    foreach ($entityTypeBundles as $entityTypeId => $bundles) {
      foreach ($bundles as $bundleId) {
        $enabledBehaviors = $this->behaviorConfigFactory->getEnabledConfiguredBehaviors($entityTypeId, $bundleId);
        if (!empty($enabledBehaviors)) {
          $result[$entityTypeId][$bundleId] = array_keys($enabledBehaviors);
        }
      }
    }

    return $result;
  }

  /**
   * Convert string to PascalCase.
   */
  private function pascalCase(string $string): string {
    return str_replace(' ', '', ucwords(str_replace('_', ' ', $string)));
  }

}
