<?php

declare(strict_types=1);

namespace Drupal\entity_type_behaviors\Plugin\GraphQLCompose\SchemaType;

use Drupal\entity_type_behaviors\EntityTypeBehaviorManager;
use Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeSchemaTypeBase;
use GraphQL\Type\Definition\ObjectType;
use GraphQL\Type\Definition\Type;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * EntityTypeBehaviorTypes GraphQL Compose Schema Type.
 *
 * @GraphQLComposeSchemaType(
 *   id = "EntityTypeBehaviorTypes",
 * )
 */
class EntityTypeBehaviorTypes extends GraphQLComposeSchemaTypeBase {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->behaviorManager = $container->get('plugin.manager.entity_type_behavior');
    return $instance;
  }

  /**
   * The behavior manager.
   *
   * @var \Drupal\entity_type_behaviors\EntityTypeBehaviorManager
   */
  private EntityTypeBehaviorManager $behaviorManager;

  /**
   * {@inheritdoc}
   */
  public function getTypes(): array {
    $types = [];
    $definitions = $this->behaviorManager->getDefinitions();

    foreach ($definitions as $definition) {
      $types[] = $this->createBehaviorType($definition);
    }

    return $types;
  }

  /**
   * Create a behavior type.
   */
  private function createBehaviorType(array $definition): ObjectType {
    $behaviorId = $definition['id'];
    $typeName = 'EntityTypeBehavior' . $this->pascalCase($behaviorId);

    $plugin = $this->behaviorManager->createInstance($behaviorId);
    $formElements = $plugin->getForm();

    $fields = [
      'behaviorType' => [
        'type' => Type::string(),
        'description' => (string) $this->t('The behavior type identifier.'),
      ],
    ];

    foreach ($formElements as $fieldName => $formElement) {
      if (!isset($formElement['#type'])) {
        continue;
      }

      $fields[$fieldName] = $this->mapFormElementToGraphqlType($fieldName, $formElement);
    }

    return new ObjectType([
      'name' => $typeName,
      'description' => (string) $this->t('Behavior settings for @behavior', ['@behavior' => $definition['label']]),
      'interfaces' => fn() => [static::type('EntityTypeBehaviorInterface')],
      'fields' => $fields,
    ]);
  }

  /**
   * Map form element to GraphQL type.
   */
  protected function mapFormElementToGraphqlType(string $fieldName, array $formElement): array {
    // Map form element types to GraphQL types.
    $baseType = match ($formElement['#type']) {
      // Add types here.
      // Boolean types.
      'checkbox' => Type::boolean(),

      // Integer types.
      'number', 'range' => Type::int(),

      // String types (explicitly listed for clarity)
      'textfield', 'textarea', 'text', 'password', 'email', 'select', 'radios' => Type::string(),

      // Default to string for any unrecognized  types.
      default => Type::string(),
    };

    // If the field is required in the form, make it non-nullable in GraphQL.
    $isRequired = isset($formElement['#required']) && $formElement['#required'] === TRUE;
    $type = $isRequired ? Type::nonNull($baseType) : $baseType;

    return [
      'type' => $type,
      'description' => isset($formElement['#description'])
        ? (string) $formElement['#description']
        : (string) $this->t('The @field value', ['@field' => $fieldName]),
    ];
  }

  /**
   * Convert string to PascalCase.
   */
  private function pascalCase(string $string): string {
    return str_replace(' ', '', ucwords(str_replace('_', ' ', $string)));
  }

}
