<?php

declare(strict_types=1);

namespace Drupal\entity_type_behaviors\Plugin\GraphQLCompose\SchemaType;

use Drupal\entity_type_behaviors\Config\BehaviorConfigFactory;
use Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeEntityTypeInterface;
use Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeSchemaTypeBase;
use GraphQL\Type\Definition\ObjectType;
use GraphQL\Type\Definition\Type;
use GraphQL\Type\Definition\UnionType;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * EntityTypeBehaviorUnions GraphQL Compose Schema Type.
 *
 * @GraphQLComposeSchemaType(
 *   id = "EntityTypeBehaviorUnions",
 * )
 */
class EntityTypeBehaviorUnions extends GraphQLComposeSchemaTypeBase {

  /**
   * The behavior configuration factory.
   *
   * @var \Drupal\entity_type_behaviors\Config\BehaviorConfigFactory
   */
  private BehaviorConfigFactory $behaviorConfigFactory;

  /**
   * Map of enabled behaviors for entity types and bundles.
   *
   * @var array
   */
  private array $enabledBehaviorsMap = [];

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->behaviorConfigFactory = $container->get('entity_type_behaviors.config.factory');
    $instance->buildEnabledBehaviorsMap();
    return $instance;
  }

  /**
   * Build a map of enabled behaviors for each entity type/bundle.
   */
  private function buildEnabledBehaviorsMap(): void {
    $entityTypeBundles = $this->getEntityTypesWithBehaviors();

    foreach ($entityTypeBundles as $entityTypeId => $bundles) {
      foreach ($bundles as $bundleId => $enabledBehaviors) {
        $key = $this->getBehaviorMapKey($entityTypeId, $bundleId);
        $this->enabledBehaviorsMap[$key] = $enabledBehaviors;
      }
    }
  }

  /**
   * Get the key for the behavior map.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   * @param string $bundleId
   *   The bundle ID.
   *
   * @return string
   *   The behavior map key.
   */
  private function getBehaviorMapKey(string $entityTypeId, string $bundleId): string {
    return "$entityTypeId:$bundleId";
  }

  /**
   * {@inheritdoc}
   */
  public function getTypes(): array {
    $types = [];
    $entityTypeBundles = $this->getEntityTypesWithBehaviors();

    foreach ($entityTypeBundles as $entityTypeId => $bundles) {
      foreach ($bundles as $bundleId => $enabledBehaviors) {
        if (empty($enabledBehaviors)) {
          continue;
        }

        $types[] = $this->createBehaviorUnionType($entityTypeId, $bundleId, $enabledBehaviors);
      }
    }

    return $types;
  }

  /**
   * Returns schema extensions for entity types with behaviors.
   *
   * Adds entity_type_behaviors field to paragraph and node entity types.
   *
   * @return array
   *   An array of schema extensions.
   */
  public function getExtensions(): array {
    $extensions = parent::getExtensions();
    $extensions = $this->addExtensionsForEntityType('paragraph', $extensions);
    $extensions = $this->addExtensionsForEntityType('node', $extensions);
    return $extensions;
  }

  /**
   * Adds extensions for a specific entity type.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   * @param array $extensions
   *   The existing extensions array.
   *
   * @return array
   *   The updated extensions array.
   */
  private function addExtensionsForEntityType(string $entityTypeId, array $extensions): array {
    $plugin = $this->gqlEntityTypeManager->getPluginInstance($entityTypeId);
    if (!$plugin instanceof GraphQLComposeEntityTypeInterface) {
      return $extensions;
    }

    return array_merge($extensions, $this->getExtensionsByPlugin($plugin, $entityTypeId));
  }

  /**
   * Gets extensions for a specific entity type plugin.
   *
   * Creates extensions for each bundle with enabled behaviors.
   *
   * @param \Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeEntityTypeInterface $plugin
   *   The entity type plugin.
   * @param string $entityTypeId
   *   The entity type ID.
   *
   * @return array
   *   An array of extensions.
   */
  public function getExtensionsByPlugin(GraphQLComposeEntityTypeInterface $plugin, string $entityTypeId): array {
    $extensions = [];
    $bundles = $plugin->getBundles();

    foreach ($bundles as $bundle) {
      $bundleEntity = $bundle->getEntity();
      $bundleId = $bundleEntity->id();

      $enabledBehaviors = $this->getEnabledBehaviorsForBundle($entityTypeId, $bundleId);
      if (empty($enabledBehaviors)) {
        continue;
      }

      $unionTypeName = 'EntityTypeBehaviors' . $this->pascalCase($entityTypeId) . $this->pascalCase($bundleId);
      $extensions[] = $this->createExtensionType($bundle->getTypeSdl(), $unionTypeName, $entityTypeId);
    }

    return $extensions;
  }

  /**
   * Creates an extension type for a bundle.
   *
   * @param string $typeName
   *   The GraphQL type name.
   * @param string $unionTypeName
   *   The union type name for behaviors.
   * @param string $entityTypeId
   *   The entity type ID.
   *
   * @return \GraphQL\Type\Definition\ObjectType
   *   The created extension type.
   */
  private function createExtensionType(string $typeName, string $unionTypeName, string $entityTypeId): ObjectType {
    return new ObjectType([
      'name' => $typeName,
      'fields' => fn() => [
        'entity_type_behaviors' => [
          'type' => Type::listOf(static::type($unionTypeName)),
          'description' => (string) $this->t('Behaviors for this @entity_type.', ['@entity_type' => $entityTypeId]),
        ],
      ],
    ]);
  }

  /**
   * Gets enabled behaviors for a specific entity type and bundle.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   * @param string $bundleId
   *   The bundle ID.
   *
   * @return array
   *   An array of enabled behaviors.
   */
  protected function getEnabledBehaviorsForBundle(string $entityTypeId, string $bundleId): array {
    return $this->behaviorConfigFactory->getEnabledConfiguredBehaviors($entityTypeId, $bundleId);
  }

  /**
   * Create a behavior union type.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   * @param string $bundleId
   *   The bundle ID.
   * @param array $enabledBehaviors
   *   The array of enabled behavior IDs.
   *
   * @return \GraphQL\Type\Definition\UnionType
   *   The created union type.
   */
  private function createBehaviorUnionType(string $entityTypeId, string $bundleId, array $enabledBehaviors): UnionType {
    $unionTypeName = 'EntityTypeBehaviors' . $this->pascalCase($entityTypeId) . $this->pascalCase($bundleId);
    $unionTypes = array_map(
      fn($behaviorId) => 'EntityTypeBehavior' . $this->pascalCase($behaviorId),
      $enabledBehaviors
    );

    return new UnionType([
      'name' => $unionTypeName,
      'description' => (string) $this->t('Entity type behaviors for @entityType:@bundle', [
        '@entityType' => $entityTypeId,
        '@bundle' => $bundleId,
      ]),
      'types' => function () use ($unionTypes) {
        return array_map(fn($typeName) => static::type($typeName), $unionTypes);
      },
      'resolveType' => function ($value) use ($entityTypeId, $bundleId) {
        $behaviorId = $value['behaviorType'] ?? NULL;
        if (!$behaviorId) {
          return NULL;
        }

        $key = $this->getBehaviorMapKey($entityTypeId, $bundleId);
        if (!isset($this->enabledBehaviorsMap[$key]) || !in_array($behaviorId, $this->enabledBehaviorsMap[$key])) {
          return NULL;
        }

        return static::type('EntityTypeBehavior' . $this->pascalCase($behaviorId));
      },
    ]);
  }

  /**
   * Get entity types with behaviors.
   *
   * @return array
   *   An associative array of entity types and their bundles
   *   with enabled behaviors.
   */
  private function getEntityTypesWithBehaviors(): array {
    $result = [];
    $entityTypeBundles = $this->behaviorConfigFactory->getConfiguredEntityTypesAndBundles();

    foreach ($entityTypeBundles as $entityTypeId => $bundles) {
      foreach ($bundles as $bundleId) {
        $enabledBehaviors = $this->behaviorConfigFactory->getEnabledConfiguredBehaviors($entityTypeId, $bundleId);
        if (!empty($enabledBehaviors)) {
          $result[$entityTypeId][$bundleId] = array_keys($enabledBehaviors);
        }
      }
    }

    return $result;
  }

  /**
   * Convert string to PascalCase.
   *
   * @param string $string
   *   The string to convert.
   *
   * @return string
   *   The PascalCase string.
   */
  private function pascalCase(string $string): string {
    return str_replace(' ', '', ucwords(str_replace('_', ' ', $string)));
  }

}
