<?php

namespace Drupal\entity_ui\EntityHandler;

use Drupal\Core\Entity\EntityHandlerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\PreloadableRouteProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Routing\RouteCollection;

/**
 * Proxy handler for entity types with a field UI, but no bundle entities.
 *
 * This hands over to one of the following handlers:
 * - \Drupal\entity_ui\EntityHandler\PlainBundlesEntityUIAdmin
 * - \Drupal\entity_ui\EntityHandler\BasicFieldUI
 *
 * These two cases can't be distinguished during entity type build, as we either
 * need to inspect routes, which can't be done during entity type build because
 * it would cause circularity.
 *
 * Cheat and don't implement EntityUIAdminInterface so we can use __call().
 */
class FieldUIWithoutBundleEntityProxy implements EntityHandlerInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The route provider service.
   *
   * @var \Drupal\Core\Routing\PreloadableRouteProviderInterface
   */
  protected $routeProvider;

  /**
   * The entity type this handler is for.
   *
   * @var \Drupal\Core\Entity\EntityTypeInterface
   */
  protected $entityType;

  /**
   * The ID of the entity type this handler is for.
   *
   * @var string
   */
  protected $entityTypeId;

  /**
   * Constructs a new EntityUIAdminBase.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Routing\PreloadableRouteProviderInterface $route_provider
   *   The route provider service.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    EntityTypeManagerInterface $entity_type_manager,
    PreloadableRouteProviderInterface $route_provider
    ) {
    $this->entityTypeId = $entity_type->id();
    $this->entityType = $entity_type;
    $this->entityTypeManager = $entity_type_manager;
    $this->routeProvider = $route_provider;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager'),
      $container->get('router.route_provider')
    );
  }

  /**
   * The real handler that this wraps.
   *
   * @var \Drupal\entity_ui\EntityHandler\EntityUIAdminInterface
   */
  protected $realHandler;

  /**
   * Override this so we instantiate the real handler.
   *
   * Routes are built before the link entities, so this is the first call to
   * this handler.
   */
  public function getRoutes(RouteCollection $route_collection) {
    $real_handler = $this->getRealHandler($route_collection);

    // We have to use the route collection we were given to prevent circularity.
    return $real_handler->getRoutes($route_collection);
  }

  /**
   * Call through to the real handler.
   */
  public function __call($name, $arguments) {
    // In any call to this handler other than getRoutes(), the router has been
    // built, so static::getRealHandler() can get the route from the route
    // provider.
    return $this->getRealHandler()->{$name}(...$arguments);
  }

  /**
   * Gets the real handler this class wraps.
   *
   * This determines the right handler to use for the entity type, by examining
   * the field UI base route.
   *
   * @param \Symfony\Component\Routing\RouteCollection|null $route_collection
   *   (optional) The route collection to get the Field UI base route from. This
   *   must be used if given, in order to prevent circularity. If omitted, the
   *   injected route provider should be used instead.
   *
   * @return \Drupal\entity_ui\EntityHandler\EntityUIAdminInterface
   *   The real handler.
   */
  protected function getRealHandler(?RouteCollection $route_collection = NULL): EntityUIAdminInterface {
    if (!isset($this->realHandler)) {
      $field_ui_base_route_name = $this->entityType->get('field_ui_base_route');

      // Get the field UI base route.
      if ($route_collection) {
        // If we've been passed a route collection, use that.
        $field_ui_base_route = $route_collection->get($field_ui_base_route_name);
      }
      else {
        $field_ui_base_route = $this->routeProvider->getRouteByName($field_ui_base_route_name);
      }

      $field_ui_base_route_path = $field_ui_base_route->getPath();

      $admin_base_route_name = NULL;
      if (!empty($field_ui_base_route_path) && str_ends_with($field_ui_base_route_path, '/{bundle}')) {
        // The entity type doesn't have a bundle entity type, but has multiple
        // bundles. These might be derived from plugins (using the Entity API
        // contrib module's functionality), or simnply hardcoded in
        // hook_entity_bundle_info(). We detect this by the presence of a
        // 'bundle' parameter at the end of the field UI route path, which Field
        // UI module expects when the bundles are not config entities.

        // We need there to be an admin base route above the parametered
        // bundle path. Remove the final '/{bundle}' path component to get its
        // path, and check it exists.
        $admin_base_path = substr($field_ui_base_route_path, 0, -(strlen('/{bundle}')));

        // Look for the admin base route in either the given collection or the
        // route provider.
        if ($route_collection) {
          foreach ($route_collection as $collection_route_name => $collection_route) {
            if ($collection_route->getPath() == $admin_base_path) {
              $admin_base_route_name = $collection_route_name;
              break;
            }
          }
        }
        else {
          $found_routes = $this->routeProvider->getRoutesByPattern($admin_base_path);
          $route_iterator = $found_routes->getIterator();
          $admin_base_route_name = $route_iterator->key();
        }
      }

      if ($admin_base_route_name) {
        // The entity type has multiple bundles, and an admin base route.
        $this->realHandler = $this->entityTypeManager->createHandlerInstance(PlainBundlesEntityUIAdmin::class, $this->entityType);

        // Set the admin base route name on the real handler, so it doesn't have
        // to repeat the work we did.
        $this->realHandler->setAdminBaseRouteName($admin_base_route_name);
      }
      else {
        // The entity type has only a single bundle, or has no admin base route.
        $this->realHandler = $this->entityTypeManager->createHandlerInstance(BasicFieldUI::class, $this->entityType);
      }
    }

    return $this->realHandler;
  }

}
