<?php

namespace Drupal\entity_ui\EntityHandler;

use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\PreloadableRouteProviderInterface;
use Symfony\Component\Routing\RouteCollection;

/**
 * Provides an admin UI for entity tabs on target entities with plain bundles.
 *
 * This is for target entity types that have multiple bundles, but where those
 * bundles are not config entities. This includes, for example, entity types
 * whose bundles are derived from plugins using the Entity API contrib module's
 * functionality, or simnply hardcoded in hook_entity_bundle_info().
 *
 * This requires the entity type to have a field UI base route defined, and for
 * that route's path to end in '/{bundle}', and for there to be a route for the
 * base of the entity type's admin UI whose path is the same as the field UI
 * path but with the '/{bundle}' component removed. So for example, an entity
 * type might define two routes with these paths:
 *  - /admin/structure/my-type
 *  - /admin/structure/my-type/{bundle}
 *
 * If either of these routes doesn't exist, the proxy handler uses BasicFieldUI
 * instead.
 *
 * The UI for entity tabs is added as a tab alongside the admin base route.
 */
class PlainBundlesEntityUIAdmin extends EntityUIAdminBase {

  /**
   * The entity type's field UI base route.
   *
   * @var string
   */
  protected $fieldUiBaseRoute;

  /**
   * The base route name for this path.
   *
   * @var string
   */
  protected $fieldUiBaseRouteName;

  /**
   * The name of the admin base route above the Field UI base route.
   *
   * Set by the proxy, which needs to determine the route prior to instantiating
   * this handler.
   *
   * @var string
   */
  protected $adminBaseRouteName;

  /**
   * Constructs a new BasicFieldUI.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Routing\PreloadableRouteProviderInterface $route_provider
   *   The route provider service.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    EntityTypeManagerInterface $entity_type_manager,
    PreloadableRouteProviderInterface $route_provider
    ) {
    parent::__construct($entity_type, $entity_type_manager, $route_provider);

    $this->fieldUiBaseRouteName = $this->entityType->get('field_ui_base_route');
  }

  /**
   * Sets the admin base route.
   *
   * @param string $admin_base_route_name
   *   The route name. Needs to work with the name rather than the route object
   *   because a route object doesn't seem to know its name, WTF.
   */
  public function setAdminBaseRouteName(string $admin_base_route_name): void {
    $this->adminBaseRouteName = $admin_base_route_name;
  }

  /**
   * {@inheritdoc}
   */
  protected function getCollectionRoutePath(RouteCollection $route_collection) {
    $field_ui_base_route = $route_collection->get($this->fieldUiBaseRouteName);

    if (empty($field_ui_base_route)) {
      throw new \Exception("Field UI base route {$this->fieldUiBaseRouteName} does not exist.");
    }

    $field_ui_base_route_path = $field_ui_base_route->getPath();
    if (!str_ends_with($field_ui_base_route_path, '{bundle}')) {
      throw new \Exception("Field UI base route {$this->fieldUiBaseRouteName} for entity type {$this->entityTypeId} must end in '/{bundle}'.");
    }

    // Remove the final '/{bundle}' path component and replace with
    // '/entity_ui'.
    $collection_route_path = substr($field_ui_base_route_path, 0, -(strlen('/{bundle}')));
    $collection_route_path .= '/entity_ui';

    return $collection_route_path;
  }

  /**
   * {@inheritdoc}
   */
  public function getLocalTasks($base_plugin_definition) {
    $tasks = [];

    // Tab for the Entity Tabs admin collection route.
    $task = $base_plugin_definition;
    $task['title'] = 'Entity tabs';
    $task['route_name'] = "entity_ui.entity_tab.{$this->entityTypeId}.collection";
    $task['base_route'] = $this->adminBaseRouteName;
    $task['weight'] = 20;

    $tasks[$task['route_name']] = $task;

    // Add a default tab for the list of bundles.
    $task = $base_plugin_definition;
    $task['title'] = t('@bundle-label list', [
      '@bundle-label' => $this->entityType->getBundleLabel(),
    ]);
    $task['route_name'] = $this->adminBaseRouteName;

    $task['base_route'] = $this->adminBaseRouteName;
    $task['weight'] = 0;

    $tasks['entity_ui.' . $this->adminBaseRouteName] = $task;

    return $tasks;
  }

}
