<?php

namespace Drupal\entity_usage_plus\Controller;

use Drupal\block_content\BlockContentInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\entity_usage\Controller\LocalTaskUsageController;
use Drupal\media\MediaInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for our pages.
 */
class LocalTaskUsagePlusController extends LocalTaskUsageController {

  /**
   * Array of entity definitions.
   *
   * @var \Drupal\Core\Entity\EntityTypeInterface[]
   */
  private array $entityTypes;

  /**
   * Array of languages.
   *
   * @var \Drupal\Core\Language\LanguageInterface[]
   */
  private array $languages;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityTypes = $instance->entityTypeManager->getDefinitions();
    $instance->languages = $instance->languageManager()->getLanguages(LanguageInterface::STATE_ALL);

    return $instance;
  }

  /**
   * Lists the usage of a given entity.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   A RouteMatch object.
   *
   * @return array
   *   The page build to be rendered.
   */
  public function listUsageLocalTask(RouteMatchInterface $route_match): array {
    $entity = $this->getEntityFromRouteMatch($route_match);
    return $this->listUsagePage($entity->getEntityTypeId(), $entity->id());
  }

  /**
   * Lists the usage of a given entity.
   *
   * @param string $entity_type
   *   The entity type.
   * @param int $entity_id
   *   The entity ID.
   *
   * @return array
   *   The page build to be rendered.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   */
  public function listUsagePage($entity_type, $entity_id): array {
    $all_rows = $this->getRows($entity_type, $entity_id);
    if (empty($all_rows)) {
      return [
        '#markup' => $this->t(
          'There are no recorded usages for entity of type: @type with id: @id',
          ['@type' => $entity_type, '@id' => $entity_id]
        ),
      ];
    }

    // Add "Relation" column.
    $header = [
      $this->t('Relation'),
      $this->t('Entity'),
      $this->t('Type'),
      $this->t('Language'),
      $this->t('Field name'),
      $this->t('Status'),
      $this->t('Used in'),
    ];

    $total = count($all_rows);
    $pager = $this->pagerManager->createPager($total, $this->itemsPerPage);
    $page = $pager->getCurrentPage();
    $page_rows = $this->getPageRows($page, $this->itemsPerPage, $entity_type, $entity_id);
    // If all rows on this page are of entities that have usage on their default
    // revision, we don't need the "Used in" column.
    $used_in_previous_revisions = FALSE;
    foreach ($page_rows as $row) {
      if (isset($row[6]['data'])) {
        $used_in = $row[6]['data'];
        $only_default = fn(array $row) => count($row) === 1 &&
          !empty($row[1]['#plain_text']) &&
          $row[1]['#plain_text'] === $this->t('Default');
        if (!$only_default($used_in)) {
          $used_in_previous_revisions = TRUE;
          break;
        }
      }
    }
    if (!$used_in_previous_revisions) {
      unset($header[6]);
      array_walk($page_rows, function (&$row, $key) {
        unset($row[6]);
      });
    }
    $build[] = [
      '#theme' => 'entity_usage_plus_usage_table',
      '#rows' => $page_rows,
      '#header' => $header,
      '#attributes' => ['class' => 'entity-usage-plus-usage-table'],
    ];

    $build[] = [
      '#type' => 'pager',
      '#route_name' => '<current>',
    ];

    return $build;
  }

  /**
   * Retrieve all usage rows for this target entity.
   *
   * @param string $entity_type
   *   The type of the target entity.
   * @param int|string $entity_id
   *   The ID of the target entity.
   *
   * @return array
   *   An indexed array of rows that should be displayed as sources or children
   *   for this target entity.
   */
  protected function getRows($entity_type, $entity_id): array {
    if (!empty($this->allRows)) {
      return $this->allRows;
      // @todo Cache this based on the target entity, invalidating the cached
      // results every time records are added/removed to the same target entity.
    }
    $rows = [];
    $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id);
    if (!$entity) {
      return $rows;
    }

    // Add the parents to the table.
    $parents = $this->getRelations($entity, 'parents');
    $rows = array_merge($rows, $parents);

    // Add a row for the current entity.
    $rows[] = [
      $this->t('Current'),
      $entity->label(),
      '---',
      '---',
      '---',
      '---',
      '---',
    ];

    // Add the children to the table.
    $children = $this->getRelations($entity, 'children');
    $rows = array_merge($rows, $children);

    $this->allRows = $rows;
    return $this->allRows;
  }

  /**
   * Get all relations for a given entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to get relations for.
   * @param string $relation
   *   The relationship type to get.
   *
   * @return array
   *   An array of rows to be displayed as relations of this entity.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function getRelations(EntityInterface $entity, string $relation = 'parents'): array {
    $rows = [];
    $revision_groups = [
      static::REVISION_DEFAULT => $this->t('Default'),
      static::REVISION_PENDING => $this->t('Pending revision(s) / Draft(s)'),
      static::REVISION_OLD => $this->t('Old revision(s)'),
    ];
    if ($relation === 'parents') {
      $relations = $this->entityUsage->listSources($entity);
    }
    else {
      if ($entity instanceof RevisionableInterface) {
        // Limit children to the current revision.
        $default_revision_id = $entity->getRevisionId();
        $relations = $this->entityUsage->listTargets($entity, $default_revision_id);
      }
      else {
        $relations = $this->entityUsage->listTargets($entity);
      }
    }

    foreach ($relations as $related_type => $ids) {
      $type_storage = $this->entityTypeManager->getStorage($related_type);
      foreach ($ids as $related_id => $records) {
        // We will show a single row per source entity. If the target is not
        // referenced on its default revision on the default language, we will
        // just show indicate that in a specific column.
        $related_entity = $type_storage->load($related_id);
        if (!$related_entity) {
          // If for some reason this record is broken, just skip it.
          continue;
        }
        // For blocks and paragraphs, get the grandchildren instead.
        $nested_types = ['paragraph', 'block_content'];
        if ($relation === 'children' && in_array($related_entity->getEntityTypeId(), $nested_types)) {
          $grandchildren = $this->getRelations($related_entity, 'grandchildren');
          if (!empty($grandchildren)) {
            $rows = array_merge($rows, $grandchildren);
          }
        }
        else {
          $default_langcode = $related_entity->language()->getId();
          if ($relation === 'parents') {
            $field_definitions = $this->entityFieldManager->getFieldDefinitions($related_type, $related_entity->bundle());
            $used_in = [];
            $revisions = [];
            if ($related_entity instanceof RevisionableInterface) {
              $default_revision_id = $related_entity->getRevisionId();
              foreach (array_reverse($records) as $record) {
                [
                  'source_vid' => $source_vid,
                  'source_langcode' => $source_langcode,
                  'field_name' => $field_name,
                ] = $record;
                // Track which languages are used in pending, default and old
                // revisions.
                $revision_group = (int) $source_vid <=> (int) $default_revision_id;
                $revisions[$revision_group][$source_langcode] = $field_name;
              }

              foreach ($revision_groups as $index => $label) {
                if (!empty($revisions[$index])) {
                  $used_in[] = $this->summarizeRevisionGroup($default_langcode, $label, $revisions[$index]);
                }
              }

              if (count($used_in) > 1) {
                $used_in = [
                  '#theme' => 'item_list',
                  '#items' => $used_in,
                  '#list_type' => 'ul',
                ];
              }
            }
            $get_field_name = function (array $field_names) use ($default_langcode, $revision_groups) {
              foreach (array_keys($revision_groups) as $group) {
                if (isset($field_names[$group])) {
                  return $field_names[$group][$default_langcode] ?? reset($field_names[$group]);
                }
              }
            };
            $field_name = $get_field_name($revisions);
          }
          else {
            // We only include children of the current revision.
            // We want to show the field name of the parent entity.
            $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity->getEntityTypeId(), $entity->bundle());
            $field_name = $records[0]['field_name'];
          }

          $link = $this->getSourceEntityLink($related_entity);
          // If the label is empty it means this usage shouldn't be shown
          // on the UI, just skip this row.
          if (empty($link)) {
            continue;
          }
          $published = $this->getSourceEntityStatus($related_entity);
          $field_label = isset($field_definitions[$field_name]) ? $field_definitions[$field_name]->getLabel() : $this->t('Unknown');
          if ($relation === 'grandchildren') {
            $parent_label = $this->entityTypes[$entity->getEntityTypeId()]->getLabel();
            $parent_label .= ': ' . $entity->bundle();
            $field_label = $parent_label . ' > ' . $field_label;
          }

          $type = $this->entityTypes[$related_type]->getLabel();
          if ($source_bundle_key = $related_entity->getEntityType()->getKey('bundle')) {
            $bundle_field = $related_entity->{$source_bundle_key};
            if ($bundle_field->getFieldDefinition()->getType() === 'entity_reference') {
              $bundle_label = $bundle_field->entity->label();
            }
            else {
              $bundle_label = $bundle_field->getString();
            }
            $type .= ': ' . $bundle_label;
          }
          $rows[] = [
            ($relation === 'parents') ? $this->t('Parent') : $this->t('-- Child'),
            $link,
            $type,
            $this->languages[$default_langcode]->getName(),
            $field_label,
            $published,
            (isset($used_in)) ? ['data' => $used_in] : '---',
          ];
        }
      }
    }

    return $rows;
  }

  /**
   * Retrieve a link to the source entity.
   *
   * Note that some entities are special-cased, since they don't have canonical
   * template and aren't expected to be re-usable. For example, if the entity
   * passed in is a paragraph or a block content, the link we produce will point
   * to this entity's parent (host) entity instead.
   *
   * @param \Drupal\Core\Entity\EntityInterface $source_entity
   *   The source entity.
   * @param string|null $text
   *   (optional) The link text for the anchor tag as a translated string.
   *   If NULL, it will use the entity's label. Defaults to NULL.
   *
   * @return \Drupal\Core\Link|string|false
   *   A link to the entity, or its non-linked label, in case it was impossible
   *   to correctly build a link. Will return FALSE if this item should not be
   *   shown on the UI (for example when dealing with an orphan paragraph).
   */
  protected function getSourceEntityLink(EntityInterface $source_entity, $text = NULL): mixed {
    // Note that $paragraph_entity->label() will return a string of type:
    // "{parent label} > {parent field}", which is actually OK for us.
    $entity_label = $source_entity->access('view label') ? $source_entity->label() : $this->t('- Restricted access -');

    $rel = NULL;
    if ($source_entity->hasLinkTemplate('revision')) {
      $rel = 'revision';
    }
    elseif ($source_entity->hasLinkTemplate('canonical')) {
      $rel = 'canonical';
    }

    // Block content likely used in Layout Builder inline or reusable blocks.
    if ($source_entity instanceof BlockContentInterface) {
      $rel = NULL;
    }

    $link_text = $text ?: $entity_label;

    // Link to media edit form.
    // This is the only change we are making to the original code function.
    if ($source_entity instanceof MediaInterface) {
      $rel = 'edit-form';
      return $source_entity->access('edit') ? $source_entity->toLink($link_text, $rel) : $link_text;
    }

    if ($rel) {
      // Prevent 404s by exposing the text unlinked if the user has no access
      // to view the entity.
      return $source_entity->access('view') ? $source_entity->toLink($link_text, $rel) : $link_text;
    }

    // Treat paragraph entities in a special manner. Normal paragraph entities
    // only exist in the context of their host (parent) entity. For this reason
    // we will use the link to the parent's entity label instead.
    /** @var \Drupal\paragraphs\ParagraphInterface $source_entity */
    if ($source_entity->getEntityTypeId() === 'paragraph') {
      $parent = $source_entity->getParentEntity();
      if ($parent) {
        return $this->getSourceEntityLink($parent, $link_text);
      }
    }
    // Treat block_content entities in a special manner. Block content
    // relationships are stored as serialized data on the host entity. This
    // makes it difficult to query parent data. Instead we look up relationship
    // data which may exist in entity_usage tables. This requires site builders
    // to set up entity usage on host-entity-type -> block_content manually.
    // @todo this could be made more generic to support other entity types with
    // difficult to handle parent -> child relationships.
    elseif ($source_entity->getEntityTypeId() === 'block_content') {
      $sources = $this->entityUsage->listSources($source_entity, FALSE);
      $source = reset($sources);
      if ($source !== FALSE) {
        $parent = $this->entityTypeManager()->getStorage($source['source_type'])->load($source['source_id']);
        if ($parent) {
          return $this->getSourceEntityLink($parent);
        }
      }
    }

    // As a fallback just return a non-linked label.
    return $link_text;
  }

}
